package org.ow2.dsrg.fm.tbpjava;

import gov.nasa.jpf.Config;
import gov.nasa.jpf.Error;
import gov.nasa.jpf.JPF;
import gov.nasa.jpf.Property;
import gov.nasa.jpf.jvm.ClassInfo;
import gov.nasa.jpf.jvm.ExceptionInfo;
import gov.nasa.jpf.jvm.MethodInfo;
import gov.nasa.jpf.jvm.NoUncaughtExceptionsProperty;
import gov.nasa.jpf.jvm.Path;
import gov.nasa.jpf.jvm.Step;
import gov.nasa.jpf.jvm.Transition;
import gov.nasa.jpf.jvm.bytecode.Instruction;
import gov.nasa.jpf.search.Search;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.io.StringWriter;
import java.lang.management.ThreadInfo;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.Map.Entry;

import javax.tools.JavaCompiler;
import javax.tools.JavaFileObject;
import javax.tools.StandardJavaFileManager;
import javax.tools.ToolProvider;

import org.ow2.dsrg.fm.tbpjava.EnvGenerator.EnvironmentDescription;
import org.ow2.dsrg.fm.tbpjava.checker.AutomatonToDot;
import org.ow2.dsrg.fm.tbpjava.checker.EventParser;
import org.ow2.dsrg.fm.tbpjava.checker.JPFSearch;
import org.ow2.dsrg.fm.tbpjava.checker.PropertyUncaughtExceptions;
import org.ow2.dsrg.fm.tbpjava.checker.ProtocolListener;
import org.ow2.dsrg.fm.tbpjava.checker.StatesMapper;
import org.ow2.dsrg.fm.tbpjava.envgen.CodeGenerator;
import org.ow2.dsrg.fm.tbpjava.envgen.StubGenerator;
import org.ow2.dsrg.fm.tbpjava.utils.CheckerResult;
import org.ow2.dsrg.fm.tbpjava.utils.ClassloaderDir;
import org.ow2.dsrg.fm.tbpjava.utils.Configuration;
import org.ow2.dsrg.fm.tbpjava.utils.DefaultJPFConfigurationGenerator;
import org.ow2.dsrg.fm.tbplib.EventTable;
import org.ow2.dsrg.fm.tbplib.EventTableImpl;
import org.ow2.dsrg.fm.tbplib.ltsa.LTSAAutomaton;
import org.ow2.dsrg.fm.tbplib.ltsa.LTSAComponentSpecification;
import org.ow2.dsrg.fm.tbplib.ltsa.NondetermisticAutomaton;
import org.ow2.dsrg.fm.tbplib.ltsa.util.ComponentReactionEdgeFiter;
import org.ow2.dsrg.fm.tbplib.parsed.ParsedComponentSpecification;
import org.ow2.dsrg.fm.tbplib.resolved.ResolvedComponentSpecification;
import org.ow2.dsrg.fm.tbplib.resolved.util.Binding;

/**
 * Public main class intended for running Checker for Threaded Behavior Protocols. 
 * @author Alfifi
 * 
 */

public class Checker {
	private static boolean JPF_WITH_TBP_CHECKER = true; // Whether to use standard JPF or JPF with TBP extension. The true is default
	
	public static final String ERROR_PREFIX = "ERROR - ";
	public static final String INTERNAL_ERROR_PREFIX = "INTERNAL ERROR - ";
	public static final String DEBUG_PREFIX  = "DEBUG - ";
	
	private static boolean DEBUG = false;
	
	/**
	 * Generates environment and runs checker.
	 */
	public static CheckerResult runTool(Configuration config, String classpathForJpfModelClasses, PrintStream out) {
		// Actualize system class path to be able load generated and compiled classes
		updateClassPath(config, out);

		/// Environment classes for each provisions
		Map<String, String> reqItfs2implClass = new TreeMap<String, String>();

		EnvGenerator generator = new EnvGenerator(config, out);
		Map<String, EnvironmentDescription> genEnvirons = generator.generateEnvironments();
		if (genEnvirons == null) {
			//Some error while parsing TBP specification
			return CheckerResult.creaResultErrorMessage("Error while generating environments. Check output to see more details.");
		}
		// Stub classes for required interface
		for(String reqItfsName : config.getComponentRequiredInterfaces().keySet()) {
			StubGenerator sb = new StubGenerator(config, reqItfsName, out);
			String generatedStubClassName = sb.generateStub(); // Generates result files
			if (generatedStubClassName == null) {
				// error occur stop execution
				return CheckerResult.creaResultErrorMessage("Error while generating environments. Check output to see more details.");
			}
			reqItfs2implClass.put(reqItfsName, generatedStubClassName);
		}
		

		// Compile generated environment classes
		JavaCompiler compiler = ToolProvider.getSystemJavaCompiler();
		if (compiler == null) {
			out.println(ERROR_PREFIX + "Cannot found java compiler to compile generated files. (Do you have JDK installed?)");
			return CheckerResult.creaResultErrorMessage("Cannot found java compiler to compile generated files. (Do you have JDK installed?)");
		}
		StandardJavaFileManager fileManager = compiler.getStandardFileManager(null, null, null);
		
		List<String> javaFileNames = new ArrayList<String>(genEnvirons.size() + reqItfs2implClass.size());
		for(String classFullName : genEnvirons.keySet()) {
			if (DEBUG) { out.println("ClassFullName=" + classFullName); }
			if (classFullName != null) {
				javaFileNames.add( config.getEnvTargetDir() + File.separator + classFullName.replace('.', '/') + ".java");
			}
		}
		for(String classFullName : reqItfs2implClass.values()) {
			javaFileNames.add( config.getEnvTargetDir() + File.separator + classFullName.replace('.', '/') + ".java");
		}
		
		
		Iterable<? extends JavaFileObject> javaFiles = fileManager.getJavaFileObjectsFromStrings( javaFileNames);

		// TODO Compiler warns about type unsafe overcasting and needless same overcasting from object to object, ...
		List<String> options = new ArrayList<String>();
		options.add("-Xlint");
		options.add("-d");
		options.add(config.getEnvTargetDir());
		options.add("-sourcepath");
		options.add(config.getEnvTargetDir());
		options.add("-cp");
		options.add(System.getProperty("java.class.path"));
		
		StringWriter compilerOutput = new StringWriter();
		Boolean compileResult = compiler.getTask(compilerOutput, fileManager, null, options, null, javaFiles).call();

		if (compileResult.booleanValue() == false) {
			out.println(ERROR_PREFIX +"Failed the compile the environment classes");
			out.println("Compilation details");
			out.println(compilerOutput.getBuffer());
			out.println(ERROR_PREFIX +"Failed the compile the environment classes");
			return CheckerResult.creaResultErrorMessage("Failed the compile the environment classes");
		}
		
		out.println("Generating automatons");
		EventTable et = new EventTableImpl();
		ParsedComponentSpecification parsedCompoSpec = generator.getComponentSpecification();
		ResolvedComponentSpecification resolvedCompSpec = parsedCompoSpec.resolve(et);
		LTSAComponentSpecification LTSACompSpec = resolvedCompSpec.makeLTSA(et);

		// Generate dot files for generated LTSAAutomatons
		DEBUG_generateDOTFilesForAutomatons(LTSACompSpec, et, config, "WithNULLs");
		// Remove null edges
		ComponentReactionEdgeFiter automatonFilter = new ComponentReactionEdgeFiter();
		automatonFilter.processSpecification(LTSACompSpec);
		DEBUG_generateDOTFilesForAutomatons(LTSACompSpec, et, config, "WithoutNULLs");
		
		
		out.println("Running patched JPF to check specification compilance");
		// Running JPF for each from generated environment
		for(String environClass : genEnvirons.keySet()) {
			EnvironmentDescription envDesc = genEnvirons.get(environClass);

			//System.out.println("Processing provision - " + environClass);
			//System.out.println("\tEnvDesc: " + envDesc.line2ProvidedCall);
			StatesMapper varStatesMap = new StatesMapper(LTSACompSpec);
			EventParser eventParser = new EventParser(config, LTSACompSpec, varStatesMap, envDesc);
			ProtocolListener pl = new ProtocolListener(config, eventParser, varStatesMap, envDesc, reqItfs2implClass, environClass, System.out);

			// Set parameters for JPF
			String envTargetDir = config.getEnvTargetDir();
			// Removing trailing '/' character
			envTargetDir = envTargetDir.substring(0, envTargetDir.length() - 1);

			 
			ArrayList<String> jpfArgs = new ArrayList<String>(10);
			DefaultJPFConfigurationGenerator.addDefaultJPFConfig(jpfArgs);
			jpfArgs.add("+search.properties= ");
			if (JPF_WITH_TBP_CHECKER) {
				jpfArgs.add("+search.class=org.ow2.dsrg.fm.tbpjava.checker.JPFSearch");
			}
			jpfArgs.add("+classpath=" + envTargetDir + ":" + System.getProperty("java.class.path"));
			jpfArgs.add("+sourcepath=" + envTargetDir + "," + converClassPathIntoJPFpath(System.getProperty("java.source.path")));
			jpfArgs.add("+report.console.property_violation=error,snapshot");
			jpfArgs.add("+vm.store_steps=true"); // To be able to analyze error trace
			if (DEBUG) {
				jpfArgs.add("+report.console.property_violation=error,trace,error,snapshot");
				jpfArgs.add("+report.console.show_method=true");
				jpfArgs.add("+report.console.show_code=true");
			}
			jpfArgs.add("+search.properties=org.ow2.dsrg.fm.tbpjava.checker.PropertyUncaughtExceptions");
			jpfArgs.add(environClass);

			String[] jpfArgsArray = new String[0];
			jpfArgsArray = jpfArgs.toArray(jpfArgsArray);

			Config jpfConfig = JPF.createConfig(jpfArgsArray);
			if (DEBUG) {
				out.println("Used JPF Configuration");
				jpfConfig.printEntries(); // Prints current configuration
			}
			JPF jpf = new JPF(jpfConfig);

			// Initialize properties (PropertyUncaughtExceptions)
			Search search = jpf.getSearch();
			if (search instanceof JPFSearch) {
				JPFSearch jpfSearch = (JPFSearch)search;
				List<Property> properties = jpfSearch.getJPFProperties();
				for(Property prop : properties) {
					if (prop instanceof PropertyUncaughtExceptions) {
						PropertyUncaughtExceptions unExProp = (PropertyUncaughtExceptions)prop;
						unExProp.initialize(config, environClass, out);
					}
				}
			}
			
			if (JPF_WITH_TBP_CHECKER) {
				jpf.addListener(pl); 
			}

			if (DEBUG) {
				// Printing used compile time configuration
				//  because of the JPFSearch that extends Search class - static method getStaticConfiguration can be called after JPF is created.
				out.println("Complile time settings");
				out.println("  Checker.JPF_WITH_TBP_CHECKER=" + JPF_WITH_TBP_CHECKER);
				out.println("  CodeGenerator ... " + CodeGenerator.getStaticConfiguration());
				out.println("  JPFSearch     ... " + JPFSearch.getStaticConfiguration());
			}

			jpf.run();
			
			if (jpf.foundErrors()) {
				Error err = jpf.getSearchErrors().get(0);
				
				CheckerResult genericErrorReport =  CheckerResult.creaResultErrorMessage("Properties violated. Check output to see more details.");
				
				if (err == null) {
					return genericErrorReport;
				}
				

				Property prop = err.getProperty();
				if (prop != null && prop instanceof PropertyUncaughtExceptions) {
					// Uncaughted exception
					PropertyUncaughtExceptions propUE = (PropertyUncaughtExceptions)prop;
					return CheckerResult.createResultCounterExample("Component throws uncaught exception.\n" /* + propUE.getErrorMessage() */, propUE.getStackTrace());
				}
				// Standard error in TBP
				Path path = err.getPath();
				if (path == null) {
					return genericErrorReport;
				}
				
				// Found last executed instruction
				Instruction lastInsn = getLastInstructionInPath(path, config);

				if (lastInsn == null) {
					return genericErrorReport;
					
				}
				MethodInfo mi = lastInsn.getMethodInfo();
				if (mi == null) {
					return genericErrorReport;
				}
				ClassInfo ci = mi.getClassInfo();
				if (ci == null) {
					return genericErrorReport;
				}
				
				
				StackTraceElement[] steArray = new StackTraceElement[1];
					steArray[0] = new StackTraceElement(ci.getName(), mi.getName(), ci.getSourceFileName(), mi.getLineNumber(lastInsn));
				
				return CheckerResult.createResultCounterExample("Counter example found.", steArray);
			}

		}
		return CheckerResult.createResultNoError();
	}

	public static CheckerResult runFromEclipse(String componentName, String componentImpl, Map<String, String> provItfs, Map<String, String> reqItfs, File protocolFile, String valueSetsClassName, File tmpDir, Collection<File> pathsToJPFModelClasses, OutputStream out)
	{
		PrintStream pOut = new PrintStream(out);
		
		List<String> parameters = new LinkedList<String>();
		
		parameters.add("component.name="+CodeGenerator.patchName(componentName));
		parameters.add("component.implclass="+componentImpl);
		
		StringBuffer strBuf1 = new StringBuffer();
		boolean first1 = true;
		for (Map.Entry<String, String> me : provItfs.entrySet())
		{
			if (first1) first1 = false;
			else strBuf1.append(";");
			strBuf1.append(me.getKey());
			strBuf1.append("-");
			strBuf1.append(me.getValue());
		}
		parameters.add("component.provitfs="+strBuf1.toString());

		StringBuffer strBuf2 = new StringBuffer();
		boolean first2 = true;
		for (Map.Entry<String, String> me : reqItfs.entrySet())
		{
			if (first2) first2 = false;
			else strBuf2.append(";");
			strBuf2.append(me.getKey());
			strBuf2.append("-");
			strBuf2.append(me.getValue());
		}
		parameters.add("component.reqitfs="+strBuf2.toString());
		parameters.add("env.protocol="+protocolFile.getAbsolutePath());
		parameters.add("env.targetdir="+tmpDir.getAbsolutePath());
		parameters.add("env.valuesets="+valueSetsClassName);
		parameters.add("env.reqitfs_setmethod=fields");
		

		Configuration config = new Configuration(parameters.toArray(new String[0]), null, pOut);
		if (config.isConfigurationError()) {
			pOut.println(ERROR_PREFIX + "Configuration contais errors.");
			return null;
		}
		
		StringBuffer strBuf3 = new StringBuffer();
		boolean first3 = true;
		for (File f : pathsToJPFModelClasses)
		{
			if (first3) first3 = false;
			else strBuf3.append(",");
			strBuf3.append(f.getAbsolutePath());
		}
		String classpathForJPFModelClasses = strBuf3.toString();
		
		return runTool(config, classpathForJPFModelClasses, pOut);
	}

	
	/**
	 * @param args Command line arguments are passed to EnvGenConfiguration object
	 */
	public static void main(String[] args) {
		if (args.length < 2) {
			System.out.println("Invalid argments number");
			System.out.println("Checker usage");
			System.out.println("  Checker configurationFile classpathForJpfModelClasses Optional-ParametersOverwriteingConfigurationFileDefaules");
			return;
		}
		
		//System.out.println("args[0] = " + args[0] + ", arvs[1] = " + args[1]);
		String configurationFile = args[0];
		String classpathForJpfModelClasses = args[1];
		String[] overwritingsParameters = new String[args.length-2];
		// Removing first parameter ... Coping args[1 .. end] into overwritingsParameters[0 ++]
		for(int i = 2; i < args.length; i++) {
			overwritingsParameters[i-2] = args[i];
		}
		
		Configuration config = new Configuration(overwritingsParameters, configurationFile, System.out);
		if (config.isConfigurationError()) {
			System.out.println("Configuration contais errors.");
			return;
		}
		
		runTool(config, classpathForJpfModelClasses, System.out);
		
	}
	
	/**
	 * Update class path and class loader to be able load classes from newly added entry.
	 * 
	 * @param config Parsed checker configuration
	 * @return true if no error occurred, false otherwise 
	 */
	private static boolean updateClassPath(Configuration config, PrintStream out) {
		String classPath = System.getProperty("java.class.path");
		classPath += File.pathSeparator + config.getEnvTargetDir();
		if (DEBUG) { out.println("java.class.path=" + classPath); }
		System.setProperty("java.class.path", classPath);
		
		ClassLoader cl = new ClassloaderDir(config.getEnvTargetDir(), Thread.currentThread().getContextClassLoader()); 
		Thread.currentThread().setContextClassLoader(cl);
		
		return true;
	}

	private static void DEBUG_generateDOTFilesForAutomatons(LTSAComponentSpecification LTSACompSpec, EventTable et, Configuration config, String fileNamePrefix) {
		Map<Binding, LTSAAutomaton> reactions = LTSACompSpec.getReactions();
		if (DEBUG) { System.out.println("Should be generated " + reactions.size() + " reaction .dot files"); }
		for(Entry<Binding, LTSAAutomaton> entry : reactions.entrySet()) {
			final Binding binding = entry.getKey();
			final LTSAAutomaton automaton = entry.getValue();
			try {
				AutomatonToDot.toDot(automaton, new FileOutputStream(config.getEnvTargetDir() + fileNamePrefix + "reaction-" + binding.getFullname() + ".dot"));
			} catch (FileNotFoundException e) {
				System.out.println("WARNING - Generating .dot file for automaton " + binding.getFullname() + " failed.");
				//e.printStackTrace();
			}
		}

		Map<String, LTSAAutomaton> threads = LTSACompSpec.getThreads();
		if (DEBUG) { System.out.println("Should be generated " + threads.size() + " threads .dot files"); }
		for( Entry<String, LTSAAutomaton> entry : threads.entrySet()) {
			final String name = entry.getKey();
			final LTSAAutomaton automaton = entry.getValue(); 
			try {
				AutomatonToDot.toDot(automaton, new FileOutputStream(config.getEnvTargetDir() + fileNamePrefix + "thread-" + name + ".dot"));
			} catch (FileNotFoundException e) {
				System.out.println("WARNING - Generating .dot file for automaton " + name + " failed.");
				//e.printStackTrace();
			}
		}
		
		
		// Note Deterministic automatons aren't processed now, (TBPLib hack) so we cann't generate .dot files 
//		Map<String, DeterministicAutomaton> provision = LTSACompSpec.getProvisions();
		Map<String, NondetermisticAutomaton> provision = LTSACompSpec.getProvisions();
		if (DEBUG) { System.out.println("Should be generated " + provision.size() + " provisions .dot files"); }
		for(Entry<String, NondetermisticAutomaton> provisionEntry : provision.entrySet()) {
			try {
				AutomatonToDot.toDot(provisionEntry.getValue(), et, new FileOutputStream(config.getEnvTargetDir() + "provision-" + provisionEntry.getKey() + ".dot"));
			} catch (FileNotFoundException e) {
				System.out.println("WARNING - Generating .dot file for automaton " + provisionEntry.getKey() + " failed.");
				//e.printStackTrace();
			}
		}
		
		
	}
	
	/**
	 * Expect file and directory list - entries are separated by {@link File#separatorChar}.
	 * If entry is file then has to end with *.jar 
	 * 
	 * Removes additional / on the end of the directory path
	 * 
	 * @param classPath Path to process. 
	 * @return File list that can by used by JPF 
	 */
	public static String converClassPathIntoJPFpath(String classPath) {
		if (classPath == null) {
			return ".";
		}
		String result = classPath.replaceAll(File.separator + File.pathSeparator, ",");
		result = result.replaceAll(File.pathSeparator, ",");
		if (result.endsWith(File.separator)) {
			result = result.substring(0, result.length() - 1);
		}
		return result;
	}
	
	
	//TODO ... skip only insctruction in generated code (co kdyz ma komponenta vic trid, ...)
	/**
	 * Gets last executed instruction in the checked component.
	 *
	 * @param path JPF execution trace to process
	 * @param config Configuration of the checker
	 * @return Gets last executed instruction in the checked component. 
	 */
	public static Instruction getLastInstructionInPath(Path path, Configuration config) {
		Instruction result = null;
		for(int i = path.size()-1; i > 0 && result == null; i--) {
			Transition tr = path.get(i);
			
			if (tr == null) {
				continue;
			}
			
			for(int j = tr.getStepCount() - 1; j > 0 && result == null; j--) {
				Step st = tr.getStep(j);
				if (st == null) {
					continue;
				}
				
				Instruction inst = st.getInstruction();
				if (inst != null) {
					MethodInfo mi = inst.getMethodInfo();
					if (mi == null) {
						continue;
					}
					ClassInfo ci = mi.getClassInfo();
					if (ci == null) {
						continue;
					}
					if (config.getComponentImplementationClasses().contains(ci.getName())) {
						result = inst;
					}
				}
			}
		}
		return result;
	}
}
