package de.itemis.qimpress.showcase.pricing_simulator.be.service;

import java.util.List;

import de.itemis.qimpress.showcase.pricing_simulator.be.domain.CustomerCategoryDiscount;
import de.itemis.qimpress.showcase.pricing_simulator.be.domain.CustomerTypeDiscount;
import de.itemis.qimpress.showcase.pricing_simulator.be.domain.ProductPrice;
import de.itemis.qimpress.showcase.pricing_simulator.be.domain.VolumeDiscount;
import de.itemis.qimpress.showcase.pricing_simulator.be.dto.Order;
import de.itemis.qimpress.showcase.pricing_simulator.be.dto.Price;
import de.itemis.qimpress.showcase.pricing_simulator.be.exceptions.ApplicationException;

/**
 * Service interface for the Pricing engine.
 * 
 * @author Wladimir Safonov
 *
 */
public interface PricingManager {

    /**
     * Loads all stored 
     * {@link de.itemis.qimpress.showcase.pricing_simulator.be.domain.VolumeDiscount VolumeDiscounts} 
     * from the database.
     * 
     * @return list of all stored 
     * {@link de.itemis.qimpress.showcase.pricing_simulator.be.domain.VolumeDiscount VolumeDiscounts}
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    public List<VolumeDiscount> getAllVolumeDiscounts() throws ApplicationException;
    
    /**
     * Loads all stored 
     * {@link de.itemis.qimpress.showcase.pricing_simulator.be.domain.CustomerCategoryDiscount CustomerCategoryDiscounts} 
     * from the database.
     * 
     * @return list of all stored 
     * {@link de.itemis.qimpress.showcase.pricing_simulator.be.domain.CustomerCategoryDiscount CustomerCategoryDiscounts}
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    public List<CustomerCategoryDiscount> getAllCustomerCategoryDiscounts() throws ApplicationException;
    
    
    /**
     * Loads all stored 
     * {@link de.itemis.qimpress.showcase.pricing_simulator.be.domain.CustomerTypeDiscount CustomerTypeDiscounts}
     * from the database.
     * 
     * @return list of all stored
     * {@link de.itemis.qimpress.showcase.pricing_simulator.be.domain.CustomerTypeDiscount CustomerTypeDiscounts}
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    public List<CustomerTypeDiscount> getAllCustomerTypeDiscounts() throws ApplicationException;
    
    /**
     * Finds a country-specific product price. If there is no country-specific price, 
     * the product standard price will be taken.
     * 
     * @param productCode code of the product to look the price for
     * @param coutryCode ISO country code to look the product price for
     * @return country-specific or standard 
     * {@link de.itemis.qimpress.showcase.pricing_simulator.be.domain.ProductPrice} 
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    public ProductPrice findProductCountryPrice(String productCode, String countryCode) throws ApplicationException;
    
    /**
     * Finds an applicable {@link de.itemis.qimpress.showcase.pricing_simulator.be.domain.VolumeDiscount}
     * for the given product quantity
     * 
     * @param productQuantity product quantity of the volume discount
     * @return applicable {@link de.itemis.qimpress.showcase.pricing_simulator.be.domain.VolumeDiscount}
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    public VolumeDiscount findVolumeDiscountByQuantity(int productQuantity) throws ApplicationException;
    
    /**
     * Finds a corresponding 
     * {@link de.itemis.qimpress.showcase.pricing_simulator.be.domain.CustomerTypeDiscount} 
     * to the given customer type.
     * 
     * @param customerType name (key) of the customer type
     * @return applicable {@link de.itemis.qimpress.showcase.pricing_simulator.be.domain.CustomerTypeDiscount}
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    public CustomerTypeDiscount findCustomerTypeDiscountByType(String customerType) throws ApplicationException;
    
    /**
     * Calculates volume-dependent price for the given product and country. Additionally applies 
     * possible discounts for the customer category and reduces the final price for reseller.
     * 
     * @param productCode code of the product to calculate the price for
     * @param countryCode ISO code of the country to look the product price for
     * @param quantity product quantity to use in volume discount rules
     * @param customerCategoryKey key of the customer category for category discounts
     * @param customerType customer type for reseller price calculation
     * @return total volume price for the product including applied discounts
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    public Price calculateVolumePriceForCustomerCategory(
            String productCode, String countryCode, int quantity, String customerCategoryKey, String customerType) 
                                                                            throws ApplicationException;
    
    /**
     * Stores all 
     * {@link de.itemis.qimpress.showcase.pricing_simulator.be.domain.VolumeDiscount VolumeDiscounts}
     * from the list to the database, updating or inserting new VolumeDiscounts as needed. Instances 
     * with existing Id will be updated, instances without Id will be inserted.
     * 
     * @param volumeDiscountsList list of 
     * {@link de.itemis.qimpress.showcase.pricing_simulator.be.domain.VolumeDiscount VolumeDiscounts}
     * to be stored
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    public void saveVolumeDiscountList(List<VolumeDiscount> volumeDiscountsList) throws ApplicationException;
    
    /**
     * Removes all given
     * {@link de.itemis.qimpress.showcase.pricing_simulator.be.domain.VolumeDiscount VolumeDiscounts}
     * in the list from the database.
     * 
     * @param volumeDiscountsList list of
     * {@link de.itemis.qimpress.showcase.pricing_simulator.be.domain.VolumeDiscount VolumeDiscounts}
     * to be deleted
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    public void deleteVolumeDiscountList(List<VolumeDiscount> volumeDiscountsList) throws ApplicationException;

    /**
     * Fills the given 
     * {@link de.itemis.qimpress.showcase.pricing_simulator.be.dto.Order}
     * with price information. Calculates
     * {@link de.itemis.qimpress.showcase.pricing_simulator.be.dto.OrderItem#getTotalPrice() price}
     * for each order item and 
     * {@link de.itemis.qimpress.showcase.pricing_simulator.be.dto.Order#getTotalPrice() total price}
     * for the order.
     * 
     * @param unpricedOrder
     * {@link de.itemis.qimpress.showcase.pricing_simulator.be.dto.Order} without price information
     * @return 
     * {@link de.itemis.qimpress.showcase.pricing_simulator.be.dto.Order} filled with price information
     * @throws ApplicationException if an error occured that cannot be resolved in the business layer
     */
    public Order priceOrder(Order unpricedOrder) throws ApplicationException;
}
