/**
 * 
 */
package eu.qimpress.ide.checkers.jpfcheck.workflow;

import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.EList;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.osgi.framework.Bundle;
import org.ow2.dsrg.fm.tbpjava.Checker;
import org.ow2.dsrg.fm.tbpjava.utils.CheckerResult;

import de.fzi.gast.types.GASTClass;
import de.fzi.gast.variables.Field;
import de.uka.ipd.sdq.workflow.IJob;
import de.uka.ipd.sdq.workflow.exceptions.JobFailedException;
import de.uka.ipd.sdq.workflow.exceptions.RollbackFailedException;
import de.uka.ipd.sdq.workflow.exceptions.UserCanceledException;
import eu.qimpress.ide.backbone.core.model.IQModel;
import eu.qimpress.ide.backbone.core.model.RepositoryModels;
import eu.qimpress.ide.checkers.jpfcheck.JPFCheckPlugin;
import eu.qimpress.ide.checkers.jpfcheck.log.Log4jOutputStream;
import eu.qimpress.ide.checkers.jpfcheck.ui.JPFCheckConfiguration;
import eu.qimpress.samm.staticstructure.ComponentType;
import eu.qimpress.samm.staticstructure.Interface;
import eu.qimpress.samm.staticstructure.InterfacePort;
import eu.qimpress.sourcecodedecorator.ComponentImplementingClassesLink;
import eu.qimpress.sourcecodedecorator.InterfaceSourceCodeLink;
import eu.qimpress.sourcecodedecorator.SourceCodeDecoratorRepository;

/**
 * Workflow job checking behavior with help of Java PathFinder (JPF).
 *
 * @author Michal Malohlava
 *
 */
public class JPFCheckJob implements IJob {
	
	private Logger logger = Logger.getLogger(JPFCheckJob.class);
	
	/* Name of a temporary directory used by JPF to generate artifacts - stored in plugin's state store */
	protected static final String SRC_GEN_DIRECTORY = "src-gen";
	
	/* Name of directory containing bundle libraries */
	protected static final String LIB_DIRECTORY = "lib/jpfcheck-bp";
	
	/* Names of jar files which have to be passed to JPF directly */ 
	protected static final String[] INTERNAL_JARS = new String[] { "jpfchecker-tool.jar" /*, /*"jpfchecker-model.jar", "jpfchecker-classes.jar"*/ };
	
	//private static final String GAST_FILE = "gastmodel.gast";
	
	//private Resource gast;

	
	private SourceCodeDecoratorRepository scdrep;
	
	/* Names of jar file which are needed for JPFCheck internal compilation task */
	protected static final String[] LIBRARIES_JARS = new String[] { 
		"jpf.jar", // Verify
		"jpf-classes.jar", // Model classes
		"jpfchecker-tool.jar"  // EnvValueSets
	};
	
	private JPFCheckConfiguration config;
	
	//private Resource sourceCodeDecoratorInstance;
	//private Resource gast;

	public JPFCheckJob(JPFCheckConfiguration config) {
		this.config = config;
	}

	/* (non-Javadoc)
	 * @see de.uka.ipd.sdq.workflow.IJob#execute(org.eclipse.core.runtime.IProgressMonitor)
	 */
	@Override
	public void execute(IProgressMonitor monitor) throws JobFailedException,
			UserCanceledException {
 
		
		logger.info("Preparing environment for JPFCheck");
		/*
		ResourceSet resset = new ResourceSetImpl(); 
		IPath path = config.getAlternative().getCorrespondingResource().getFullPath();
		URI uri =  URI.createPlatformResourceURI(path.toOSString(), true);
		gast = resset.getResource(uri.appendSegment(GAST_FILE), true);
		*/
		/* Temporary directory */
		File tempDir = getTemporaryDir(config.getProject());
		
		/* References to internal jars required by JPF */
		Collection<File> internalJarFiles = getInternalJarFiles(INTERNAL_JARS);
		internalJarFiles.add(tempDir);		
		/* References to application binaries stored in workspace directory */		
		Collection<File> appBinaries = getAppBinaries(config.getAlternative().getRepository().getQProject().getProject());
		appBinaries.add(tempDir);
		
		/* ----------------------------------- */		
		// Setup classloder
		logger.debug(" - modifying context classloader for current thread");
		Thread currentThread = Thread.currentThread();
		final ClassLoader oldClassloader = currentThread.getContextClassLoader();
		try {
			Collection<URL> urls = new LinkedList<URL>();
			for (File f : internalJarFiles) {
				urls.add(f.toURI().toURL());
			}
			for(File f: appBinaries) {
				urls.add(f.toURI().toURL());
			}
			urls.add(tempDir.toURI().toURL());
			
			ClassLoader newClassLoader = new URLClassLoader(urls.toArray(new URL[urls.size()]), oldClassloader);
				
			currentThread.setContextClassLoader(newClassLoader);
			
		} catch (Exception e) {
			e.printStackTrace();
		}
		
		// prepare class path for JPF
		logger.debug(" - modifying env. var. java.class.path for JPF");
		StringBuffer sb = new StringBuffer(System.getProperty("java.class.path"));
		for (File f : getInternalJarFiles(LIBRARIES_JARS)) {
			sb.append(File.pathSeparatorChar);
			sb.append(f.getAbsolutePath());
		}
		
		for (File f : appBinaries) {
			sb.append(File.pathSeparatorChar);
			sb.append(f.getAbsolutePath());			
		}
		
		System.setProperty("java.class.path", sb.toString());
		
		// TODO delete created markers
		// TODO delete output console
		PrintStream oldOut = System.out;
		PrintStream oldErr = System.err;
		PrintStream osInfo = null;
		PrintStream osDebug = null;
		try {	
			//we need the decorator to get the real interface names
			IQModel sourceCodeDecoratorInstance = config.getAlternative().getModel(RepositoryModels.SOURCECODE_DECORATOR_EXTENSION);
			 
			scdrep = sourceCodeDecoratorInstance.getTopLevelEObject(SourceCodeDecoratorRepository.class);
			EList<InterfaceSourceCodeLink> codeLinks = scdrep.getInterfaceSourceCodeLink();
			
			Map<String, String> provIfaces = getIfacesMap(config.getComponentType().getProvided(), codeLinks);
			Map<String, String> reqIfaces = getRequiredIfacesMap(config.getComponentType(), codeLinks);
			
			// setup outputs
			osInfo = new PrintStream(new Log4jOutputStream(logger, Level.INFO, oldClassloader));
			osDebug = new PrintStream(new Log4jOutputStream(logger, Level.DEBUG, oldClassloader));
						
			System.setOut(osInfo);
			System.setErr(osDebug);
		
			printAnalysisParameters(config.getComponentType().getName(), config.getComponentImpl(), 
					provIfaces, reqIfaces, config.getProtocolFile(), 
					config.getAdditionalParameters().get("env.valuesets"), 
					tempDir, internalJarFiles, osDebug);
			
			logger.info("Launching JPFCheck tool");
			
			CheckerResult counterExample = Checker.runFromEclipse(
					config.getComponentType().getName(), 
					config.getComponentImpl(), 
					provIfaces, 
					reqIfaces, 
					config.getProtocolFile(), 
					config.getAdditionalParameters().get("env.valuesets"), 
					tempDir, 
					internalJarFiles, 
					osInfo
					);

			try {
				refreshTempFolder();
			} catch (CoreException ex) {
				logger.warn("Cannot refresh generated folder " + SRC_GEN_DIRECTORY);
			}
			
			// process the result
			processResults(counterExample);
			
		} catch (Throwable e) {
			e.printStackTrace();
			
			logger.error("Exception occured during launching JPFChecker", e);
			
			final Throwable e2 = e;
			
			PlatformUI.getWorkbench().getDisplay().asyncExec(new Runnable() {
				@Override
				public void run() {
					ErrorDialog.openError(PlatformUI.getWorkbench().getDisplay().getActiveShell(), 
							"JPFCheck error", 
							"Error occured during calling JPFCheck tool!", 
							new Status(IStatus.ERROR, JPFCheckPlugin.PLUGIN_ID, "JPFCheck exception", e2));
				}
				
			}
				
			);
			
		} finally {			
			
			// close streams
			System.setOut(oldOut);
			System.setErr(oldErr);
			osDebug.close();
			osInfo.close();
			
			// set original classloader
			currentThread.setContextClassLoader(oldClassloader);			
		}
	}

	/* (non-Javadoc)
	 * @see de.uka.ipd.sdq.workflow.IJob#getName()
	 */
	@Override
	public String getName() {		
		return "JPFCheck job";
	}

	/* (non-Javadoc)
	 * @see de.uka.ipd.sdq.workflow.IJob#rollback(org.eclipse.core.runtime.IProgressMonitor)
	 */
	@Override
	public void rollback(IProgressMonitor monitor)
			throws RollbackFailedException {
		// TODO do cleanup
	}
	
	/**
	 * Return reference to temporary directory. If the directory does not exist, it is created.
	 * 
	 * @param project project including temporary directory
	 * 
	 * @return temporary directory
	 * @throws JobFailedException is thrown when the temporary directory cannot be established
	 */
	protected File getTemporaryDir(IProject project) throws JobFailedException {
		File tempDir = null;
		
		try {
			IFolder srcGenFolder = project.getFolder(SRC_GEN_DIRECTORY);
			if (!srcGenFolder.exists()) {
				srcGenFolder.create(true, true, null);
				project.refreshLocal(0, null);
			}
				
			tempDir = Platform.getLocation().append(srcGenFolder.getFullPath()).toFile();
		} catch (CoreException e) {
			throw new JobFailedException(e);
		}
				 
		return tempDir;		
	}
	
	/**
	 * Return a collection of references to internal jars stored in bundle.
	 * 
	 * Note: bundle has to be deployed unpacked to received concrete files!
	 * 
	 * @param jarNames names of jar which are required
	 * 
	 * @return collection of files representing bundles jars.
	 */
	protected Collection<File> getInternalJarFiles(String[] jarNames) {
		Bundle thisBundle = JPFCheckPlugin.getDefault().getBundle();
		Collection<File> result = new LinkedList<File>();
		
		for (String jarName : jarNames) {
						
			try {
				URL url = thisBundle.getEntry("/" + LIB_DIRECTORY + "/" +  jarName);
				if (url != null) {
					result.add(new File(FileLocator.toFileURL(url).getPath()));
				}
			} catch (IOException e) {				
				e.printStackTrace();
			} 
		}
				
		return result;
	}

	public void setConfig(JPFCheckConfiguration config) {
		this.config = config;
	}

	public JPFCheckConfiguration getConfig() {
		return config;
	}
	
	private Map<String, String> getIfacesMap(EList<InterfacePort> ifacePorts, EList<InterfaceSourceCodeLink> codelinks) {
		Map<String, String> result = new HashMap<String, String>();
		
		
		for (InterfacePort ifacePort : ifacePorts) {
			for (InterfaceSourceCodeLink link : codelinks)
				if (link.getInterface().getName().equals(ifacePort.getInterfaceType().getName())) {
					result.put(link.getGastClass().getQualifiedName(), link.getGastClass().getQualifiedName());
					break;
				}
		}
		
		return result;		
	}
	

	private Map<String, String> getRequiredIfacesMap(ComponentType ctype, EList<InterfaceSourceCodeLink> codelinks) {
		EList<InterfacePort> ports = ctype.getRequired();
        Map<String, String> result = new HashMap<String, String>();
        for (InterfacePort port : ports) {
            Interface i = port.getInterfaceType();
            String iname = i.getName();
            GASTClass implIface = null;
            
            for (InterfaceSourceCodeLink link : scdrep.getInterfaceSourceCodeLink()) {
            	if (link.getInterface().getName().equals(iname)) {
            		implIface = link.getGastClass();
            		break;
            	}
            }
            
            //here go through gast implementing classes and find an (!!!) appropriate field.
            //Set<GASTClass> classes = implClasses.get(ctype.getName());
            EList<ComponentImplementingClassesLink> classes = scdrep.getComponentImplementingClassesLink();
            
            for (ComponentImplementingClassesLink links : classes) {
            	if (links.getComponent().getName().equals(ctype.getName())) {
	            	for (GASTClass c : links.getImplementingClasses()) {
		            	for (Field f : c.getFields()) {
		            		if (f.getType().getQualifiedName().equals(implIface.getQualifiedName())) {
		            			iname = f.getSimpleName();
		            			result.put(iname, implIface.getQualifiedName());
		            		}
		            	}
	            	}
            	}
            
             }
        }
		return result;		
	}
	
	 
	private List<File> getAppBinaries(IProject project) {
		List<File> result = new LinkedList<File>();
		IWorkspaceRoot wRoot = ResourcesPlugin.getWorkspace().getRoot();
		
		try {
			IJavaProject javaProject = JavaCore.create(project);
			IPath entryPath = null;
			
			entryPath = javaProject.getOutputLocation();
			result.add(new File(wRoot.getFile(entryPath).getLocationURI()));
			
			IClasspathEntry[] cpEntries = javaProject.getResolvedClasspath(true);
			for (IClasspathEntry cpEntry : cpEntries) {
				File file = null;
				entryPath = cpEntry.getPath();
				switch (cpEntry.getEntryKind()) {
				
				case IClasspathEntry.CPE_SOURCE:
				case IClasspathEntry.CPE_LIBRARY:
					if (entryPath.toFile().exists()) {
						file = entryPath.toFile();
					} else {
						IFile ifile = wRoot.getFile(entryPath);
						if (ifile != null) {					
							file = new File(wRoot.getFile(entryPath).getLocationURI());
						}
					}					
					break;
					
				case IClasspathEntry.CPE_CONTAINER:					
				case IClasspathEntry.CPE_PROJECT:							
				case IClasspathEntry.CPE_VARIABLE:
					break;
				}
				
				if (file != null)
					result.add(file);
			}
		} catch (JavaModelException e) {
			e.printStackTrace();
		}
		
		
		return result;		
	}
	
	protected void printAnalysisParameters(String componentName, String componentImpl, Map<String, String> provIfaces, Map<String, String> reqIfaces, File protocolFile, String valueSetsClassName, File tempDir, Collection<File> internalJarFiles, PrintStream msgStream) {
		msgStream.println("JPFCheck tool");
		msgStream.println("------------------");
		msgStream.println("Input parameters:");
		msgStream.println("  componentName: " + componentName);
		msgStream.println("  componentImpl: " + componentImpl);
		msgStream.println("  provided ifaces:");
		for(Map.Entry<String, String> e : provIfaces.entrySet()) {
			msgStream.println("    " + e.getKey() + " : " + e.getValue());									
		}
		msgStream.println("  required ifaces:");
		for(Map.Entry<String, String> e : reqIfaces.entrySet()) {
			msgStream.println("    " + e.getKey() + " : " + e.getValue());									
		}
		msgStream.println("  protocolFile: " + protocolFile);
		msgStream.println("  valueSetsClassName: " + valueSetsClassName);
		msgStream.println("  tempDir: " + tempDir);
		msgStream.println("  internal JARS:" );
		for(File f : internalJarFiles) {
			msgStream.println("    " + f);
		}
		msgStream.println("------------------");
	}
	
	protected void processResults(final CheckerResult checkerResult) {
		if (checkerResult == null) {
			return;
		}

		final Display display = PlatformUI.getWorkbench().getDisplay();
		
		display.asyncExec( new Runnable() {
			@Override
			public void run() {
				// Three case - no error, error with no thace, error with trace
				if (checkerResult.isError() == false) {
					MessageDialog.openInformation(
							PlatformUI.getWorkbench().getDisplay().getActiveShell(), 
							"JPFChecker result", 
							checkerResult.getMessage());
					return;
				}

				// Any error takes place
				if (checkerResult.getStackTrace() != null) {
					try {
						List<IMarker> markers = createMarkers(checkerResult.getStackTrace());
						if (markers != null && markers.size() > 0) {
							IDE.openEditor(PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage(), markers.get(0));
						}
					} catch (Throwable t) {
						t.printStackTrace();
						System.err.println(t.getMessage());
					}
					ErrorDialog.openError(
							display.getActiveShell(), 
							"JPFChecker result", 
							"JPFChecker found counter example", 
							new JPFCheckCIStatus(checkerResult.getMessage(), checkerResult.getStackTrace()));
				} else {
					ErrorDialog.openError(
							display.getActiveShell(), 
							"JPFChecker result", 
							"JPFChecker found counter example", 
							new Status(IStatus.ERROR, JPFCheckPlugin.PLUGIN_ID, IStatus.OK, checkerResult.getMessage(), null));
				}
				
			}
		});
	}
	
	private List<IMarker> createMarkers(StackTraceElement[] elements) throws CoreException {
		IProject project = config.getProject();
		Collection<IFolder> folders = new LinkedList<IFolder>();
		List<IMarker> resultingMarkers = new LinkedList<IMarker>();
		
		for (IResource r : project.members() ) {
			if (r.getType() == IResource.FOLDER) {
				folders.add((IFolder) r);
			}
		}
		
		for (StackTraceElement element : elements) {
			for (IFolder folder : folders) {
				IPath stPath = new Path(element.getFileName());
				IResource res = folder.findMember(stPath.segment(stPath.segmentCount()-1));
				if (res != null && res.getType() == IResource.FILE && res.exists() ) {
					// TODO it would be better to register own marker type
					res.refreshLocal(0, null);
					IMarker marker = res.createMarker(IMarker.TEXT);
					marker.setAttribute(IMarker.LINE_NUMBER, element.getLineNumber());
					marker.setAttribute(IDE.EDITOR_ID_ATTR, "org.eclipse.ui.JavaEditor");
					resultingMarkers.add(marker);
					
					break;
				}
			}
		}
		
		return resultingMarkers;
	}
	
	protected void refreshTempFolder() throws CoreException {
		IProject project = config.getProject();
		
		IFolder folder = project.getFolder(SRC_GEN_DIRECTORY);
		if (folder != null && folder.exists()) {
			folder.refreshLocal(IResource.DEPTH_INFINITE, null);
		}
	}
}
