/**
 * 
 */
package eu.qimpress.ide.checkers.jpfcheck.log;

import java.io.IOException;
import java.io.OutputStream;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;

/**
 * @author Michal Malohlava
 * 
 *         Based on the comments in discussions
 *         http://www.mail-archive.com/log4j
 *         -user%40logging.apache.org/msg05005.html
 * 
 */
public class Log4jOutputStream extends OutputStream {

	public static final int DEFAULT_BUFFER_LENGTH = 2048;
	
	protected static final String LINE_SEPARATOR = System.getProperty("line.separator");

	private ClassLoader classLoader;

	private Logger logger;

	private Level level;

	/* Buffer */
	private boolean hasBeenClosed = false;
	private byte[] buffer;
	private int bufferLength;
	private int count;

	public Log4jOutputStream(Logger logger, Level level, final ClassLoader classLoader) {
		this.logger = logger;
		this.level = level;
		this.classLoader = classLoader;

		bufferLength = DEFAULT_BUFFER_LENGTH;
		buffer = new byte[bufferLength];
		count = 0;
	}

	public Log4jOutputStream(Logger logger, Level level) {
		this(logger, level, null);
	}

	@Override
	public void write(int b) throws IOException {
		if (hasBeenClosed) {
			throw new IOException("The stream has been closed.");
		}

		if (b == 0) {
			return;
		}
		
		if (count == bufferLength) {
			int newBufferLength = bufferLength + DEFAULT_BUFFER_LENGTH;
			byte[] newBuffer = new byte[newBufferLength];
			
			System.arraycopy(buffer, 0, newBuffer, 0, bufferLength);
			buffer = newBuffer;
			bufferLength = newBufferLength;
		}
		
		buffer[count++] = (byte) b;
		
		// flush after each line
		if (bufferEndsWithNewline()) {
			flush();
		}
	}

	@Override
	public void flush() throws IOException {
		if (count == 0) {
			return;
		}
		
		// don't print out blank lines; flushing from PrintStream puts out these
	    if (count == LINE_SEPARATOR.length()) {
	      if ( bufferEndsWithNewline() ) {
	        reset();
	        return;
	      }
	    }
	    
		int lenToOutput = 0;
		if (bufferEndsWithNewline()) {
			lenToOutput = count - LINE_SEPARATOR.length();			
		} else {
			lenToOutput = count;
		}
		
		byte[] outputBuffer = new byte[lenToOutput];
		
		System.arraycopy(buffer, 0, outputBuffer, 0, lenToOutput);
		
		ClassLoader cl = null;
		if (classLoader != null) {
			cl = switchClassLoader(classLoader);
		}
		logger.log(level, new String(outputBuffer));
		if (classLoader != null) {
			switchClassLoader(cl);
		}
		
		reset();		
	}
	
	protected boolean bufferEndsWithNewline() {
		if (count >= LINE_SEPARATOR.length()) {
			if ( (LINE_SEPARATOR.length() == 1 
					&& ((char) buffer[count-1]) == LINE_SEPARATOR.charAt(0)
				  )
					||
				 (LINE_SEPARATOR.length() == 2 
						 && ((char) buffer[count-1]) == LINE_SEPARATOR.charAt(1) 
						 && ((char) buffer[count-2]) == LINE_SEPARATOR.charAt(0)
				 )) {
				return true;
			}
		}
		
		return false;		
	}
	
	protected ClassLoader switchClassLoader(final ClassLoader cloader) {
		ClassLoader cl = Thread.currentThread().getContextClassLoader();
		Thread.currentThread().setContextClassLoader(cloader);
		return cl;
	}
	
	private void reset() {
		count = 0;		
	}

	@Override
	public void close() throws IOException {
		flush();
		hasBeenClosed = true;
	}

}
