package eu.qimpress.ide.backbone.core.model;

import java.util.List;

import eu.qimpress.ide.backbone.core.operations.SaveQModelUIOperation;
import eu.qimpress.resultmodel.AlternativeEvaluation;
import eu.qimpress.resultmodel.ResultRepository;

/**
 * Interface for accessing result model.
 * 
 * @author Michal Malohlava
 * @author Viliam Simko
 */
public interface IQResultRepositoryAccess {

	/**
	 * Returns a representation of result model. If the result model does not
	 * exist in the project, the model is created.
	 * 
	 * @return I-Q representation of result model, always returns not-null
	 * @throws RepositoryException 
	 */
	IQModel getResultModel() throws RepositoryException;

	/**
	 * Returns an instance of {@link ResultRepository} associated with this
	 * project.
	 * 
	 * If the repository does not exist new result model including repository is
	 * created.
	 * 
	 * The model needs to be saved by the user - see {@link IQModel#save()} and workspace operation {@link SaveQModelUIOperation}.
	 * 
	 * @return returns instance of {@link ResultRepository}, always returns not-null
	 * @throws RepositoryException 
	 */
	ResultRepository getResultRepository() throws RepositoryException;

	/**
	 * Create a new {@link AlternativeEvaluation} in the
	 * {@link ResultRepository}.
	 * 
	 * Returns a new instance of {@link AlternativeEvaluation}. The caller has
	 * to explicitly save the result model including this newly created
	 * {@link AlternativeEvaluation}.
	 * 
	 * The model needs to be saved by the user - see {@link IQModel#save()} and workspace operation {@link SaveQModelUIOperation}.
	 * 
	 * @return new {@link AlternativeEvaluation}, always returns not-null
	 * @throws RepositoryException 
	 */
	AlternativeEvaluation createAlternativeEvaluation() throws RepositoryException;

	/**
	 * Create new {@link AlternativeEvaluation} with initial alternativeId in the project ResultModel.
	 * 
	 * The model needs to be saved by the user - see {@link IQModel#save()} and workspace operation {@link SaveQModelUIOperation}. 
	 * 
	 * @param alternativeId
	 * @return new {@link AlternativeEvaluation}, always returns not-null
	 * @throws RepositoryException 
	 */
	AlternativeEvaluation createAlternativeEvaluation(String alternativeId) throws RepositoryException;
	
	/**
	 * Deletes an alternative evaluation by searching through the whole
	 * repository.
	 * 
	 * @param alternativeEvaluation
	 * @throws RepositoryException 
	 */
	void deleteAlternativeEvaluation(AlternativeEvaluation alternativeEvaluation) throws RepositoryException;

	/**
	 * Returns a list of {@link AlternativeEvaluation} connected to given
	 * alternative.
	 * 
	 * @param id
	 *            identifier of alternative
	 * @return a list of {@link AlternativeEvaluation}
	 * @throws RepositoryException 
	 */
	List<AlternativeEvaluation> getAlternativeEvaluationsByAlternativeId(
			String id) throws RepositoryException;

	/**
	 * Returns all {@link AlternativeEvaluation} included in the
	 * {@link ResultRepository}.
	 * 
	 * @return a list of {@link AlternativeEvaluation}
	 * @throws RepositoryException 
	 */
	List<AlternativeEvaluation> getAllAlternativeEvaluations() throws RepositoryException;
}
