package eu.qimpress.ide.backbone.core.model;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspace;


/**
 * Q-ImPrESS application model abstraction.
 * 
 * The application model consists of Q-ImPrESS projects (i.e., projects with
 * Q-ImPrESS nature) which further includes alternative repositories.
 * 
 * @author Michal Malohlava
 * 
 */
public interface IQApplicationModel extends IQElement, IParent {
	/**
	 * Get a list of current workspace Q-ImPrESS projects.
	 * 
	 * @return projects with Q-ImPrESS nature.
	 */
	IQProject[] getQProjects();

	/**
	 * Get Q-ImPrESS project abstraction related to a given project.
	 * 
	 * @param project
	 *            workspace project
	 * 
	 * @return return related project abstraction or null if project does not
	 *         exist
	 */
	IQProject getQProject(IProject project);
	
	/**
	 * Remove given project from application model.
	 * 
	 * @param project project to be removed
	 */
	void removeQProject(IProject project) throws RepositoryException;

	/**
	 * Get current workspace.
	 * 
	 * @return the current workspace
	 */
	IWorkspace getWorkspace();
	
	/**
	 * Adds the given listener for change events to this application model. Has
	 * no effect if an identical listener is already registered.
	 * 
	 * @param listener the listener
	 */
	void addChangeListener(IQApplicationModelChangeListener listener);
	
	/**
	 * Removes an already registered listener.
	 * 
	 * @param listener the listener
	 */
	void removeChangeListener(IQApplicationModelChangeListener listener);
	/**
	 * Notifies all attached listeners that a refresh is needed.
	 */
	void fireRefresh();
	
	/**
	 * Notifies all attached listeners that a new alternative has been created.
	 * 
	 * @param alternative created alternative
	 */
	void fireAlternativeCreated(IQAlternative alternative);
	
	/**
	 * Notifies all attached listeners that a new set of model has been created.
	 * 
	 * @param models created models
	 */
	void fireModelCreated(IQModel[] models);
	
	/**
	 * Notifies about modification models.
	 * 
	 * @param models set of modified models.
	 */
	void fireModelModified(IQModel[] models);
	
	/**
	 * Notifies listeners about model deletion
	 * 
	 * @param models set of deleted models
	 */
	void fireModelDeleted(IQModel[] models);
	
	/**
	 * Search the IQApplication model for the IQElemnt with the specified ID
	 * @param id The ID of the IQElement to retrieve
	 * @return The element with the given ID or null if the element does not exist in this IQApplication
	 */
	IQElement getElementByID(String id);
}
