/* $I1d$ */
package eu.qimpress.ide.backbone.core.internal.model;

import java.lang.reflect.InvocationTargetException;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.neodatis.odb.ODB;
import org.neodatis.odb.ODBFactory;
import org.neodatis.odb.Objects;
import org.neodatis.odb.core.query.IQuery;
import org.neodatis.odb.core.query.criteria.Where;
import org.neodatis.odb.impl.core.query.criteria.CriteriaQuery;

import eu.qimpress.ide.backbone.core.model.IQAlternative;
import eu.qimpress.ide.backbone.core.model.IQAlternativeInfo;
import eu.qimpress.ide.backbone.core.model.IQInitializer;
import eu.qimpress.ide.backbone.core.model.IQModel;
import eu.qimpress.ide.backbone.core.model.IQProject;
import eu.qimpress.ide.backbone.core.model.IQRepository;
import eu.qimpress.ide.backbone.core.model.IQWorkspaceController;
import eu.qimpress.ide.backbone.core.model.ISaveable;
import eu.qimpress.ide.backbone.core.model.RepositoryException;
import eu.qimpress.ide.backbone.core.model.RepositoryModels;
import eu.qimpress.resultmodel.AlternativeEvaluation;
import eu.qimpress.resultmodel.ResultModelFactory;
import eu.qimpress.resultmodel.ResultRepository;

/**
 * Implementation of the
 * {@link eu.qimpress.ide.backbone.core.model.IQRepository} interface. Stores
 * data in a directory &mdash; each alternative in a separated subdirectory. 
 * 
 * Cache the list of alternatives and update the cache after each alternative operation.
 * 
 * @author Michal Malohlava
 */
public class QCachedDirectoryRepositoryImpl extends QElement implements IQRepository,
		ISupportSaveable {

	/** Logger. */
	final static Logger logger = Logger.getLogger(QCachedDirectoryRepositoryImpl.class);
	final static IQAlternative[] EMPTY_ALTERNATIVE_ARRAY = new IQAlternative[] {};
	
	/** Default repository directory. */
	public static final String DEFAULT_REPOSITORY_LOCATION = "alternatives";
	
	/** Default location of the repository database. */
	public static final String DEFAULT_REPOSITORY_DB_LOCATION = ".db";
	
	private QCachedDirectoryRepositoryController controller;

	/** Directory of the repository. */
	private IFolder directory;
	
	/** Parent project for this repository. */
	private IQProject qProject;	
	/** Cached alternatives */
	private Map<String, IQAlternative> alternativeCache = new LinkedHashMap<String, IQAlternative>();	
	private IQAlternative defaultAlternative = null;
	private IQAlternative globalAlternative = null;
	
	/** Simple handle to a database file which needs to be locally refreshed */
	/* package */ IFile dbFile;
	/* package */ String dbFileName;
		
	/**
	 * Constructor for repository.
	 * 
	 * The repository is not initialized by default. Explicit call of the {@link #init()} method is required.
	 * If the repository is not initialized the methods calls throws an exception. 
	 * 
	 * @param dir directory of the repository
	 */
	/* package */ QCachedDirectoryRepositoryImpl(IQProject project, final IFolder directory)
			throws RepositoryException {
		super(project);
		
		this.qProject = project;
		this.directory = directory;
		this.dbFile = this.directory.getFile(DEFAULT_REPOSITORY_DB_LOCATION);
		this.dbFileName = this.dbFile.getLocation().toOSString();
		
		this.controller = new QCachedDirectoryRepositoryController(this);
	}	
	
	/**
	 * Get ODB database handle.
	 * 
	 * @return
	 */
	private ODB getODB() {
		return ODBFactory.open(this.dbFileName);		
	}
	
	/**
	 * Release ODB db handle.
	 * 
	 * @param odb ODB handle
	 */
	private void releaseODB(ODB odb) {
		odb.close();
		odb = null;

		// refresh the DB file
		try {
			WorkspaceModifyOperation operation = new WorkspaceModifyOperation(this.directory) {				
				@Override
				protected void execute(IProgressMonitor monitor) throws CoreException,
						InvocationTargetException, InterruptedException {					
					dbFile.refreshLocal(IResource.DEPTH_ZERO, null);					
				}
			};	
			operation.run(null);
			
		} catch (Exception e) {
			logger.warn("Cannot refresh database file", e);
		}
	}
	
	/* package */ void doCacheAlternatives() throws RepositoryException {
		synchronized (this.dbFile) {
			ODB odb = getODB();
			// cache all alternatives
			try {
				Objects<QAlternativeInfoImpl> obs = odb.getObjects(QAlternativeInfoImpl.class);
				
				updateRetrievedAlternatives(obs);
			} finally {
				releaseODB(odb);
			}
		}
	}
	
	private void updateRetrievedAlternatives(
			Objects<QAlternativeInfoImpl> obs)
			throws RepositoryException {
		
		for (QAlternativeInfoImpl ai : obs) {
			IQAlternative alternative = new QAlternativeImpl(this, null, ai);
			
			if (!alternative.getCorrespondingResource().exists()) {
				// TODO delete such alternative from the database
				logger.warn("Alternative does not exist. Skipped. AlternativeInfo: " + ai + ", repository: " + this);
				continue;
			}
			
			if (ai.getId().equals(GLOBAL_ALTERNATIVE_ID)) {
				this.globalAlternative = alternative;				
			} else {
				alternativeCache.put(alternative.getInfo().getId(), alternative);
			}
			if (ai.isDefault()) {
				this.defaultAlternative = alternative;
			}
		}
		
		// update parent-child relation
		for (IQAlternative alternative : alternativeCache.values()) {
			if (alternative.getInfo().getParent() == null) {
				((QElement) alternative).parent = this;				
			} else {
				((QElement) alternative).parent = alternativeCache.get(alternative.getInfo().getParent().getId());
			}
		}
		
		// NOTE the updateRetrievedAlternatives does not create the underlying resource - see controller
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IQAlternative[] listAllAlternatives() throws RepositoryException {
		return alternativeCache.values().toArray(new IQAlternative[alternativeCache.size()]);		
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @throws RepositoryException
	 */
	@Override
	public IQAlternative[] listTopLevelAlternatives()
			throws RepositoryException {
				
		List<IQAlternative> result = new ArrayList<IQAlternative>(4);
		for (IQAlternative alt : alternativeCache.values()) {
			if (alt.getInfo().getParent() == null) {
				result.add(alt);								
			}
		}
		
		Collections.sort(result, QAlternativeComparator.INSTANCE);
		
		return result.toArray(new IQAlternative[result.size()]);		
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @throws RepositoryException
	 */
	@Override
	public IQAlternative getAlternative(String id) throws RepositoryException {
		if (id.equals(GLOBAL_ALTERNATIVE_ID)) {
			return this.globalAlternative;
		} else {		
			return alternativeCache.containsKey(id) ? alternativeCache.get(id) : null;
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @throws RepositoryException
	 */
	@Override
	public IQAlternative[] getChildren(IQAlternative parentAlt)
			throws RepositoryException {
		
		List<IQAlternative> result = new ArrayList<IQAlternative>(4);
		
		for (IQAlternative alt : alternativeCache.values()) {
			if (alt.getParent() != null && alt.getParent().equals(parentAlt) ) {
				result.add(alt);								
			}
		}
		
		Collections.sort(result, QAlternativeComparator.INSTANCE);
		
		return result.toArray(new IQAlternative[result.size()]);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IQAlternative createAlternative(String description)
			throws RepositoryException {
		return createAlternative(null, description, false, null);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IQAlternative createAlternative(IQAlternative parent,
			String description) throws RepositoryException {
		return createAlternative(parent, description, false);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IQAlternative createAlternative(IQAlternative parent,
			String description, boolean noCopy) throws RepositoryException {
		return createAlternative(parent, description, noCopy, null);
	}
	
	public IQAlternative createAlternative(String description, 
			String specifiedId)	throws RepositoryException{
		return createAlternative(null, description, false, specifiedId);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public IQAlternative createAlternative(IQAlternative parent,
			String description, boolean noCopy, String specifiedId) 
			throws RepositoryException {		
		IQAlternativeInfo parentInfo = parent != null ? parent.getInfo() : null;

		// manage alternative tree
		IQAlternativeInfo alternativeInfo = null; 		
		
		if (specifiedId != null){
			alternativeInfo = new QAlternativeInfoImpl(parentInfo, description, specifiedId);
		} else {
			alternativeInfo = new QAlternativeInfoImpl(parentInfo, description);
		}

		// save alternative info
		synchronized (this.dbFile) { // should be synchronized									
			ODB odb = getODB();
			try {
				odb.store(alternativeInfo);
				odb.commit();			
			} finally {
				releaseODB(odb);
			}
		}
		
		IQAlternative alt = new QAlternativeImpl(this, parent, alternativeInfo);
		IQWorkspaceController altController = (IQWorkspaceController) (alt.getAdapter(IQWorkspaceController.class));		
		altController.init(false, true);				
		
		// put alternative into a cache
		if (alternativeInfo.getId().equals(GLOBAL_ALTERNATIVE_ID)) {
			this.globalAlternative = alt;
		} else {
			alternativeCache.put(alt.getInfo().getId(), alt);
		}
		
		return alt;
	}
	

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void deleteAlternative(IQAlternative alternative)
			throws RepositoryException {
		deleteAlternative(alternative, false, false);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void deleteAlternative(IQAlternative alternative,
			boolean deleteContent, boolean force) throws RepositoryException {		
		IQAlternative[] children = getChildren(alternative);
		if (force) {
			for (IQAlternative chAlt : children) {
				deleteAlternative(chAlt, deleteContent, force);
			}
		} else { // do not delete children
			if (children.length != 0) {
				throw new RepositoryException(
						"Cannot be deleted - there are child alternatives");
			}
		}
		
		synchronized(this.dbFile) {			
			ODB odb = getODB();
			try {
				String id = alternative.getInfo().getId();
				IQuery q = new CriteriaQuery(QAlternativeInfoImpl.class, Where.equal("id", id));
				Objects<QAlternativeInfoImpl> obs = odb.getObjects(q);
				
				if (obs.size() != 0) {
					odb.delete(obs.getFirst());			
					odb.commit();					
				}				
			} finally {
				releaseODB(odb);
			}
		}
		// remove alternative from cache
		alternativeCache.remove(alternative.getInfo().getId());
		
		// FIXME should be in a workspace operation
		if (deleteContent) {
			try {
				if (alternative.getAlternativeFolder().exists()) {
					alternative.getAlternativeFolder().delete(true, false, null);
				}
			} catch (CoreException ex) {
				throw new RepositoryException(
						"Cannot delete content of the alternative \""
								+ alternative.getInfo().getId() + "\"", ex);
			}
		}
	}
	
	@Override
	public void deleteAlternativeEvaluation(
			AlternativeEvaluation alternativeEvaluation)
			throws RepositoryException {

		getResultRepository().getAnalysisRuns().remove(alternativeEvaluation);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void close() throws RepositoryException {
		// DO nothing in this implementation
	}

	@Override
	public boolean isClosed() {
		return false;
	}
	
	@Override
	public ElementType getElementType() {
		return ElementType.Q_REPOSITORY;
	}

	@Override
	public IResource getCorrespondingResource() {
		return getRepositoryFolder();
	}

	@Override
	public IFolder getRepositoryFolder() {
		return this.directory;
	}

	@Override
	public IQAlternative getDefaultAlternative() throws RepositoryException {
		return this.defaultAlternative;
	}
	
	/* Fix for bug 251 from JIRA, http://jira.ow2.org/browse/QIMPRESS-251 */
	private void unsetNondefaultAlternatives(IQAlternative newDefAlt)
		throws RepositoryException {
		
		IQRepository repo = newDefAlt.getRepository();
		IQAlternative[] alternatives = repo.listAllAlternatives();
		
		for (IQAlternative alt : alternatives){
			if (alt != newDefAlt){
				alt.getInfo().setDefault(false);
				save((QAlternativeInfoImpl) newDefAlt.getInfo());
			}
		}
	}
	
	@Override
	public void setDefaultAlternative(IQAlternative newDefAlt)
			throws RepositoryException {
		
		unsetNondefaultAlternatives(newDefAlt);
		
		if (this.defaultAlternative == newDefAlt) {
			return;			
		}
		
		IQAlternative oldDefAlt = this.defaultAlternative;		
		newDefAlt.getInfo().setDefault(true);
		this.defaultAlternative = newDefAlt;		
		
		if (oldDefAlt != null) {
			oldDefAlt.getInfo().setDefault(false);			
			save((QAlternativeInfoImpl) oldDefAlt.getInfo(), (QAlternativeInfoImpl) newDefAlt.getInfo());		
		} else {
			save((QAlternativeInfoImpl) newDefAlt.getInfo());			
		}
	}
	
	@Override
	public IQAlternative getGlobalAlternative() throws RepositoryException {
		
		IQAlternative globalAlternative = null;
		
		try {
			globalAlternative = getAlternative(GLOBAL_ALTERNATIVE_ID);
		} catch (RepositoryException e) {
			logger.debug("Exception occured during getting global alternative", e);		
		}

		if (globalAlternative == null && this.controller.isReady()) {
			throw new RepositoryException("Global alternative not found, there should always be a single global alternative in a project. To avoid this exception disable and enable Q-I nature for project: " + this.getQProject().getProject().getName());
		}
		
		return globalAlternative;
	}

	@Override
	public void save(ISaveable o) throws RepositoryException {
		synchronized (this.dbFile) {
			ODB odb = getODB();
			try {
				odb.store(o);
				odb.commit();
			} finally {
				releaseODB(odb);
			}
		}
	}
	
	protected void save(ISaveable ... os) throws RepositoryException {
		synchronized (this.dbFile) {									
			ODB odb = getODB();
			try {
				for (ISaveable o : os) {
					odb.store(o);
				}			
				odb.commit();
			} finally {
				releaseODB(odb);
			}
		}
	}

	@Override
	public IQProject getQProject() {
		return (IQProject) getParent();
	}

	/* ========== IQParameterProvider interface methods ======== */
	@Override
	public boolean containsKey(String key) {
		boolean result = false;
		
		synchronized(this.dbFile) {
			ODB odb = getODB();
			try {
				CriteriaQuery query = new CriteriaQuery(QRepositoryParam.class, Where
					.equal("key", key));
				result = odb.count(query).compareTo(new BigInteger("0")) > 0;
			} finally {
				releaseODB(odb);
			}
		}
		
		return result; 		
	}

	@Override
	public Object getParameter(String key) {
		IQuery query = new CriteriaQuery(QRepositoryParam.class, Where.equal(
				"key", key));
		
		Objects<Object> objects = null;
		synchronized(this.dbFile) {
			ODB odb = getODB();
			try {
				objects = odb.getObjects(query);
			} finally {
				releaseODB(odb);
			}
		}
		
		if (objects.size() == 0)
			return null;

		return ((QRepositoryParam) objects.getFirst()).value;		
	}

	@Override
	public void setParameter(String key, Object value) {
		IQuery query = new CriteriaQuery(QRepositoryParam.class, Where.equal(
				"key", key));
		
		synchronized(this.dbFile) {
			ODB odb = getODB();
			try {
				Objects<Object> objects = odb.getObjects(query);
				if (objects.size() == 0) {
					if (value != null)
						odb.store(new QRepositoryParam(key, value));
				} else {
					QRepositoryParam param = (QRepositoryParam) objects.getFirst();
					if (value == null)
						odb.delete(param);
					else {
						param.value = value;
						odb.store(param);
					}
				}
		
				odb.commit();
			} finally {			
				releaseODB(odb);
			}
		} /* end of synchronized */		
	}

	/* ========== IQResultRepositoryAccess interface methods ======== */

	@Override
	public AlternativeEvaluation createAlternativeEvaluation() throws RepositoryException {		
		AlternativeEvaluation newAlternativeEvaluation = ResultModelFactory.eINSTANCE.createAlternativeEvaluation();
		// store create AlternativeEvalutation into result repository -> it needs explicit save
		getResultRepository().getAnalysisRuns().add(newAlternativeEvaluation);
		
		return newAlternativeEvaluation;
	}
	
	@Override
	public AlternativeEvaluation createAlternativeEvaluation(String alternativeId) throws RepositoryException {
		AlternativeEvaluation newAltEval = createAlternativeEvaluation(); 
		newAltEval.setAlternativeId(alternativeId);
		
		return newAltEval;
	}

	@Override
	public List<AlternativeEvaluation> getAllAlternativeEvaluations() throws RepositoryException {
		return getResultRepository().getAnalysisRuns();
	}

	@Override
	public List<AlternativeEvaluation> getAlternativeEvaluationsByAlternativeId(String id) throws RepositoryException {
		
		// aggregated list of matching items
		List<AlternativeEvaluation> listFound = new LinkedList<AlternativeEvaluation>();
		
		for (AlternativeEvaluation altEval : getAllAlternativeEvaluations()) {
			
			// filter on alternativeId
			if(id.equals( altEval.getAlternativeId() ))
				listFound.add(altEval);
		}
		
		return listFound;
	}

	@Override
	public IQModel getResultModel() throws RepositoryException {
		IFile resultModelFile = getResultRepositoryFile();
		if (resultModelFile != null) {
			IQModel qModel = new QModelImpl(resultModelFile, getGlobalAlternative());
			return qModel;
		} else {
			return null;
		}
	}
	
	@Override
	public ResultRepository getResultRepository() throws RepositoryException {

		IQModel resultModel = getResultModel();

		ResultRepository resultRepository = resultModel.getTopLevelEObject(
				ResultRepository.class, ResultModelFactory.eINSTANCE
						.getResultModelPackage().getResultRepository());

		return resultRepository;
	}
	
	/**
	 * Returns a URI that represents a valid default result repository XMI file.
	 * The XMI file is located in the "alternatives" directory within the
	 * project.
	 * 
	 * Note: I couldn't find any better way of converting a filename to the URI.
	 * Feel free to correct this implementation if necessary.
	 */
	protected IFile getResultRepositoryFile() {
		
		try {
			IFolder defAlternativeFolder = getGlobalAlternative().getAlternativeFolder();
			return defAlternativeFolder.getFile(RepositoryModels.GLOBAL_REPOSITORY_MODEL_NAME + "."
					+ RepositoryModels.RESULT_MODEL_EXT);
		} catch (Exception e){
			logger.warn("Cannot get result repository file!", e);
			return null;
		}
	}
	
	/**
	 * Returns a URI that represents a valid default usage model XMI file.
	 * The XMI file is located in the "alternatives" directory within the
	 * project.
	 * 
	 * Note: I couldn't find any better way of converting a filename to the URI.
	 * Feel free to correct this implementation if necessary.
	 */
	protected IFile getUsageModelFile() {
		
		try {
			IFolder defAlternativeFolder = qProject.getRepository().getGlobalAlternative().getAlternativeFolder();
			return defAlternativeFolder.getFile(RepositoryModels.USAGE_MODEL_NAME + "."
					+ RepositoryModels.USAGE_MODEL_EXT);
		} catch (Exception e){
			logger.warn("Cannot get usage model file!", e);
			return null;
		}
	}	

	@Override
	public IQModel getGlobalModel(String modelType) throws RepositoryException {		
		IQAlternative alternative = getGlobalAlternative();
								
		return alternative.getModel(modelType); 
	}
	
	@Override
	public Object getAdapter(Class adapter) {
	
		if (adapter.equals(IQWorkspaceController.class)) {
			return this.controller;
		} else if (adapter.equals(IQInitializer.class)) {
			return this.controller;
		}
		
		return super.getAdapter(adapter);
	}
	
	private static final class QAlternativeComparator implements Comparator<IQAlternative> {
		private static final QAlternativeComparator INSTANCE = new QAlternativeComparator();

		@Override
		public int compare(IQAlternative o1, IQAlternative o2) {		
			String s1 = o1.getInfo().getDescription();
			String s2 = o2.getInfo().getDescription();
			return s1.compareTo(s2);
		}
		
	}
}
