/**
 * 
 */
package eu.qimpress.ide.backbone.core.internal.model;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;

import eu.qimpress.ide.backbone.core.QImpressCore;
import eu.qimpress.ide.backbone.core.QImpressNature;
import eu.qimpress.ide.backbone.core.model.IQAlternative;
import eu.qimpress.ide.backbone.core.model.IQApplicationModel;
import eu.qimpress.ide.backbone.core.model.IQApplicationModelChangeListener;
import eu.qimpress.ide.backbone.core.model.IQElement;
import eu.qimpress.ide.backbone.core.model.IQModel;
import eu.qimpress.ide.backbone.core.model.IQProject;
import eu.qimpress.ide.backbone.core.model.IQRepository;
import eu.qimpress.ide.backbone.core.model.RepositoryException;

/**
 * @author Michal Malohlava
 *
 */
public class QApplicationModelImpl extends QElement implements IQApplicationModel {
	
	private Map<IProject, IQProject> projectsMap = new HashMap<IProject, IQProject>();
	
	private Set<IQApplicationModelChangeListener> refreshListeners = new HashSet<IQApplicationModelChangeListener>();
	
	private static final Logger logger = Logger.getLogger(QApplicationModelImpl.class);
	
	public QApplicationModelImpl() {
		super(null);
		
		// initialize application model 
		initApplicationModel();
	}
	
	/**
	 * Initializes application model - gets workspace and contained projects and creates appropriate IQProject. 	
	 */
	private void initApplicationModel() {
		IWorkspace workspace = getWorkspace();		
		IProject[] projects = workspace.getRoot().getProjects();
		
		for(IProject project : projects) {
			if (hasQINature(project)) {
				// ignore return value
				// IQProject qProject = 
				getQProject(project);
			}
		}
	}

	@Override
	public ElementType getElementType() {		
		return ElementType.Q_APPLICATION_MODEL;
	}

	@Override
	public IResource getCorrespondingResource() {		
		return null;
	}

	@Override
	public synchronized IQProject getQProject(IProject project) {
		if (project == null) {
			return null;
		}
		
		if (!project.isAccessible()) {				
			return null;
		}
		
		if (projectsMap.containsKey(project)) {
			return projectsMap.get(project);
		} else {	
			IQProject qProject = new QProjectImpl(project, this);
			projectsMap.put(project, qProject);
			
			return qProject;
		}
	}

	@Override
	public IQElement[] getChildren() {
		return getQProjects();
	}

	@Override
	public boolean hasChildren() {		
		return !projectsMap.isEmpty();
	}

	@Override
	public IWorkspace getWorkspace() {
		return ResourcesPlugin.getWorkspace();
	}

	@Override
	public IQProject[] getQProjects() {		
		Collection<IQProject> qProjects = projectsMap.values();  
		
		return qProjects.toArray(new IQProject[qProjects.size()]);
	}

	@Override
	public void addChangeListener(IQApplicationModelChangeListener listener) {
		refreshListeners.add(listener);
	}

	@Override
	public void removeChangeListener(IQApplicationModelChangeListener listener) {
		refreshListeners.remove(listener);
	}

	@Override
	public void fireRefresh() {
		for (IQApplicationModelChangeListener listener : refreshListeners) {
			listener.refresh();
		}
	}

	@Override
	public void fireAlternativeCreated(IQAlternative alternative) {
		for (IQApplicationModelChangeListener listener : refreshListeners) {
			listener.alternativeCreated(alternative);
		}
	}
	
	@Override
	public void fireModelCreated(IQModel[] models) {
		updateModelsInAlternatives(models); // fixed BUG #206
		for (IQApplicationModelChangeListener listener : refreshListeners) {
			listener.modelCreated(models);
		}
	}
	
	@Override
	public void fireModelModified(IQModel[] models) {
		updateModelsInAlternatives(models); // fixed BUG #206
		for (IQApplicationModelChangeListener listener : refreshListeners) {
			listener.modelModified(models);
		}
	}
		
	@Override
	public void fireModelDeleted(IQModel[] models) {
		updateModelsInAlternatives(models); // fixed BUG #206
		for (IQApplicationModelChangeListener listener : refreshListeners) {
			listener.modelDeleted(models);
		}
		
	}
	
	/**
	 * Triggers update of models within affected alternatives.
	 * Fixes bug #206
	 * @param models List of affected models
	 * @author Viliam Simko
	 */
	public void updateModelsInAlternatives(IQModel[] models) {
		for (IQModel iqModel : models) {
			
			// deriving alternative from each affected model
			IPath loc = iqModel.getCorrespondingResource().getLocation();
			String alternativeId = QImpressCore.getAlternativeId(loc);
			IQAlternative alternative = QImpressCore.getAlternativeById(alternativeId);
			
			// models will be refreshed from the filesystem
			alternative.getModels(true);
		}
	}
	
	@Override
	public IQElement getElementByID(String id) {
		
       for (IQProject p : projectsMap.values()) {
			try {
				IQRepository repo = p.getRepository();
				if (!repo.isClosed()) {
					IQAlternative alt = p.getRepository().getAlternative(id);
					if (alt != null) {
						return alt;
					}
				}
			} catch (RepositoryException ex) {
				logger.debug("Cannot get element by ID: " + id);
			} 
		}
		return null;
	}

	private boolean hasQINature(IProject project) {
		if (project.isAccessible()) {
			try {
				return project.hasNature(QImpressNature.NATURE_ID);
			} catch (CoreException e) {
				logger.debug("Cannot get project nature for project " + project);
			}
		}
		
		return false;		
	}

	@Override
	public void removeQProject(IProject project) throws RepositoryException {
		if (projectsMap.containsKey(project)) {
			IQProject qProject = projectsMap.get(project);
			
			qProject.getRepository().close();
			projectsMap.remove(project);						
		} else {
			logger.warn("Trying to remove projects which is not in application model, project: " + project);
		}
	}
	
}
