/*
 * 
 * 
 */
package eu.qimpress.ide.backbone.core.internal.model;

import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;

import eu.qimpress.ide.backbone.core.model.IQAlternative;
import eu.qimpress.ide.backbone.core.model.IQAlternativeInfo;
import eu.qimpress.ide.backbone.core.model.IQElement;
import eu.qimpress.ide.backbone.core.model.IQModel;
import eu.qimpress.ide.backbone.core.model.IQRepository;
import eu.qimpress.ide.backbone.core.model.IQWorkspaceController;
import eu.qimpress.ide.backbone.core.model.ISaveable;
import eu.qimpress.ide.backbone.core.model.RepositoryException;
import eu.qimpress.ide.backbone.core.model.RepositoryModels;

/**
 * Alternative implementation.
 * 
 * @author Petr Hnetynka
 * @author Michal Malohlava
 */
public class QAlternativeImpl extends QElement implements IQAlternative, ISaveable {

	private final static Logger logger = Logger.getLogger(QAlternativeImpl.class);
	
	public final static String DEFAULT_MODEL_NAME = "model";	
	public final static String TEMPORARY_FOLDER_NAME = "tmp";
	
	private IFolder alternativeFolder;
	private IQAlternativeInfo alternativeInfo;
	private IQRepository repository;
	private IQAlternative parentAlternative;
	
	/**
	 * Models map.
	 */
	private Map<String, IQModel> models = null;
	
	private QAlternativeController controller;
	
	QAlternativeImpl(IQRepository repository, IQElement parent, final IQAlternativeInfo alternativeInfo) throws RepositoryException {
		super(parent == null ? repository : parent);
		
		this.alternativeInfo = alternativeInfo;
		this.alternativeFolder = repository.getRepositoryFolder().getFolder(alternativeInfo.getId());
		this.repository = repository;
		this.parentAlternative = getParentAlternative();
		this.controller = new QAlternativeController(this);
	}
	
	protected final IQAlternative getParentAlternative() {
		IQAlternativeInfo parentInfo = this.getInfo().getParent();
		
		if (parentInfo != null) {
			IQAlternative parentAlternative = null;
			try {
				parentAlternative = this.repository.getAlternative(parentInfo.getId());
			} catch (RepositoryException e) { /* ignored */ logger.warn("Cannot get the parent alternative for " + this); }
			
			return parentAlternative; 			
		} else {
			return null;
		}				
	}
	
	protected void initModels() {		
		
		if (models == null) {
			models = new HashMap<String, IQModel>();
		}
		
		models.clear();
		
		try {
			IResource[] members = alternativeFolder.members();
			for (IResource member : members) {
				if (member.exists() && member.getType() == IResource.FILE) {
					if (RepositoryModels.isSupportedFileExtension(member.getFullPath().getFileExtension())) {
						IQModel model = new QModelImpl((IFile) member, this);
						this.models.put(model.getType(), model);
					}
				}
			}
		} catch (CoreException e) {
			logger.error("Cannot initizalie alternative's models.", e);
		}
	}

	@Override
	public ElementType getElementType() {
		return ElementType.Q_ALTERNATIVE;
	}

	@Override
	public IFolder getAlternativeFolder() {		
		return this.alternativeFolder;
	}

	@Override
	public IQAlternativeInfo getInfo() {		
		return this.alternativeInfo;
	}

	/** {@inheritDoc}
	 */
	@Override
    public boolean isTopLevel() {
      return getInfo().getParent() == null;
    }

	@Override
	public IQModel getModel(String name) throws RepositoryException {
		if (models == null) {
			initModels();
		}
		
		IQModel model = models.get(name);
		
		if (model == null) {
			model = createNewModel(DEFAULT_MODEL_NAME, name);
			models.put(name, model);
		}
		
		return model;		
	}
	
	/**
	 * Create new Q-ImPrESS model but it is not stored on the filesystem!
	 * 
	 * @param name name of model (in fact file extension)
	 * @return new model
	 * 
	 */
	protected IQModel createNewModel(String name, String type) {
		String filename = name + "." + type;
		
		IFile modelFile = alternativeFolder.getFile(filename);		
		
		IQModel qModel = new QModelImpl(modelFile, this);
		
		return qModel;
	}

	@Override
	public IQModel[] getModels() {
		return getModels(false);		
	}
	
	@Override
	public IQModel[] getModels(boolean refreshModelCache) {
		if (models == null || refreshModelCache) {
			initModels();
		}
		
		return models.values().toArray(new IQModel[models.values().size()]);
	}

	@Override
	public IQModel getStaticStructureModel() throws RepositoryException {		
		return getModel(RepositoryModels.REPOSITORY_MODEL_EXT);
	}

	@Override
	public Iterator<IQModel> iterator() {		
		return models.values().iterator();
	}

	@Override
	public IResource getCorrespondingResource() {		
		return this.alternativeFolder;
	}

	@Override
	public IQRepository getRepository() {		
		return this.repository;
	}
	
	@Override
	public IQModel storeEObject(EObject eo, String filename, String modelType) throws RepositoryException {
		// FIXME re-implement this via adapter to Resource.class
		String modelFilename = filename + "." + modelType;		
		IFile modelFile = alternativeFolder.getFile(modelFilename);
		ResourceSet rSet = new ResourceSetImpl();
	
		IQModel qModel = null;
		try {
			qModel = saveModelResource(rSet, eo, modelFile);
			
			if (models != null) {
				models.put(modelType, qModel); // FIXME dispatch event about creation new model
			}
		} catch (IOException e) {			
			logger.error("Error occured during saving model into a file", e);
			
			throw new RepositoryException(e);
		}
		
		return qModel;
	}
	
	@Override
	public IQModel[] storeEObject(EObject[] eos, String[] filenames, String[] modelTypes) throws RepositoryException {
		assert(eos.length == filenames.length && eos.length == modelTypes.length && filenames.length == modelTypes.length);
				
		IQModel[] result = new IQModel[eos.length];
		
		ResourceSet rSet = new ResourceSetImpl();
		
		try {
			for(int i = 0; i < eos.length; i++) {
				EObject eo = eos[i];
				String filename = filenames[i];
				String modelType = modelTypes[i];
				String modelFilename = filename + "." + modelType;
				IFile modelFile = alternativeFolder.getFile(modelFilename);
				
				IQModel qModel = saveModelResource(rSet, eo, modelFile);
				
				result[i] = qModel;
			}
		} catch (IOException e) {
			logger.error("Error occured during saving model into a file", e);
			
			throw new RepositoryException(e);			
		}
		
		// cache model files
		if (models != null) {
			for (int i = 0; i < result.length; i++) {
				if (result[i] != null) {
					models.put(result[i].getType(), result[i]);					
				}
			}
		}
		
		return result;
	}
	
	protected IQModel saveModelResource(ResourceSet rSet, EObject eo, IFile modelFile) throws IOException {
		URI fileURI = URI.createPlatformResourceURI(modelFile.getFullPath().toOSString(), false);
		Resource resource = rSet.createResource(fileURI);
		
		resource.getContents().add(eo);	
		resource.save(Collections.EMPTY_MAP);
		
		IQModel qModel = new QModelImpl(modelFile, this);
		
		// this would be notified by workspace listener
		//QImpressApplicationModelManager.getManager().getQAppModel().fireModelCreated(new IQModel[] { qModel });
		
		return qModel;		
	}
	
	// ISaveable interface implementation
	@Override
	public void save() throws RepositoryException {
		if (this.repository instanceof ISupportSaveable &&
			this.alternativeInfo instanceof ISaveable) {
			((ISupportSaveable) this.repository).save((ISaveable) this.alternativeInfo);
		}
	}
		

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((alternativeInfo == null) ? 0 : alternativeInfo.hashCode());
		result = prime
				* result
				+ ((parentAlternative == null) ? 0 : parentAlternative
						.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		QAlternativeImpl other = (QAlternativeImpl) obj;
		if (alternativeInfo == null) {
			if (other.alternativeInfo != null)
				return false;
		} else if (!alternativeInfo.equals(other.alternativeInfo))
			return false;
		if (parentAlternative == null) {
			if (other.parentAlternative != null)
				return false;
		} else if (!parentAlternative.equals(other.parentAlternative))
			return false;
		return true;
	}

	@Override
	public String toString() {
		return "QAlternativeImpl [alternativeInfo=" + alternativeInfo + "]";
	}
	
	@Override
	public void deleteModel(String name) throws RepositoryException {
		QModelImpl m = (QModelImpl) models.get(name);
		if (m != null) {		  		  
		  m.internalDelete();
		  internalDelete(name);
		} else {
			throw new RepositoryException("No model with the given name.");
		}
	}
	
	/** Removes the model from the internal structures.
	 * 
	 */
	void internalDelete(String modelName) {
		models.remove(modelName);
	}

	@Override
	public IFolder getTemporaryFolder() throws CoreException {
		IFolder altFolder = getAlternativeFolder();
		
		IFolder result = altFolder.getFolder(TEMPORARY_FOLDER_NAME);
		if (result.exists()){
			/* nothing to be done */
		} else {
			result.create(false, true, null);
		}
		return result;
	}	
	
	@Override
	public Object getAdapter(Class adapter) {
		if (adapter.equals(IQWorkspaceController.class)) {
			return this.controller;
		}
		
		return super.getAdapter(adapter);
	}
}
