/**
 * 
 */
package eu.qimpress.ide.backbone.core.internal.model;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.Iterator;

import org.apache.log4j.Logger;
import org.eclipse.core.internal.resources.ResourceException;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.ui.actions.WorkspaceModifyOperation;

import eu.qimpress.ide.backbone.core.model.IQAlternative;
import eu.qimpress.ide.backbone.core.model.RepositoryModels;
import eu.qimpress.samm.staticstructure.InterfacePort;
import eu.qimpress.samm.staticstructure.Operation;
import eu.qimpress.samm.usagemodel.SystemCall;

/**
 * Controller for {@link IQAlternative} entity.
 *
 * @author Michal Malohlava
 *
 */
public class QAlternativeController extends AbstractQController<QAlternativeImpl> {
	
	private static final Logger logger = Logger.getLogger(QAlternativeController.class);

	public QAlternativeController(QAlternativeImpl element) {
		super(element);
	}
	
	@Override
	protected void doInit(IProgressMonitor monitor) throws CoreException, InvocationTargetException, InterruptedException {
		WorkspaceModifyOperation operation =getInitWSOperation();		
		operation.run(monitor);
	}
	
	@Override
	public WorkspaceModifyOperation getInitWSOperation() {		
		return new CreateAlternativeOperation();
	}
		
	private class CreateAlternativeOperation extends WorkspaceModifyOperation {
		
		private IFile serviceArchitectureModel = null;
		private IFile usageModel = null;
		private IFile repositoryModel = null;
		
		public CreateAlternativeOperation() {
			// lock the directory with alternatives
			super(element.getCorrespondingResource().getParent());
		}
		
		@Override
		protected void execute(IProgressMonitor monitor) throws CoreException,
				InvocationTargetException, InterruptedException {
			IFolder alternativeFolder = element.getAlternativeFolder();
			
			if (!alternativeFolder.exists()) {
				alternativeFolder.create(true, true, monitor);				
				
				if (element.getParentAlternative() != null) {
					IResource[] parentContent = element.getParentAlternative().getAlternativeFolder().members();
	                  alternativeFolder.getWorkspace().copy(parentContent, alternativeFolder.getFullPath(), true, monitor);
	                  
	                  handleEMFReferences(alternativeFolder, monitor);
				}
			}
		}

		@SuppressWarnings("restriction")
		private void handleEMFReferences(final IFolder alternativeFolder, IProgressMonitor monitor) throws CoreException {
			
			alternativeFolder.accept(new IResourceVisitor() {
				
				@Override
				public boolean visit(IResource resource) throws CoreException {
					if (resource.getType() == IResource.FILE) {						
						if (resource.getName().endsWith(RepositoryModels.SERVICE_ARCHITECTURE_MODEL_EXT)) {
							serviceArchitectureModel = (IFile) resource;							
						} else if (resource.getName().endsWith(RepositoryModels.USAGE_MODEL_EXT)) {
							usageModel = (IFile) resource;							
						} else if (resource.getName().endsWith(RepositoryModels.REPOSITORY_MODEL_EXT)) {
							repositoryModel = (IFile) resource;
						}
					}
					
					return true;
				}
			}, IResource.DEPTH_ONE, false);
			
			// service architecture model exists => it could be referenced
			if (serviceArchitectureModel != null && usageModel != null) {
				// handle references
				handleUsageModel(usageModel, serviceArchitectureModel, repositoryModel);
			}
		}		
		
		@SuppressWarnings("restriction")
		private void handleUsageModel(final IFile usageModel, final IFile serviceArchitectureModel, IFile repositoryModel) throws CoreException {
			ResourceSet rset = new ResourceSetImpl();
			
			Resource rUsageModel = rset.getResource(getModelURI(usageModel), true);
			Resource rServiceArchModel = rset.getResource(getModelURI(serviceArchitectureModel), true);
			Resource rRepositoryModel = rset.getResource(getModelURI(repositoryModel), true);
			
			TreeIterator<EObject> it = EcoreUtil.getAllProperContents(rUsageModel, true);
			while (it.hasNext()) {
				EObject eo = it.next();
				
				if (eo instanceof SystemCall) {
					SystemCall systemCall = (SystemCall) eo;
					String ifacePortId = systemCall.getCalledInterfacePort().getId();
					InterfacePort ifacePort = getInterfacePortById(ifacePortId, rServiceArchModel);
					// this is just a work-around to fix wrong models
					if (ifacePort == null) {
						ifacePort = getInterfacePortById(ifacePortId, rRepositoryModel);
						logger.debug("InterfacePort found in repository and not in service architecture. Hence, the model looks wrong!");
					}
					
					if (ifacePort!=null) {
						systemCall.setCalledInterfacePort(ifacePort);
					} else {
						systemCall.setCalledInterfacePort(null);
						logger.debug("Cannot re-set ifaceport reference for systemcall="+systemCall);
						
					}
					
					String operationId = systemCall.getOperation().getId();
					Operation operation = ifacePort!=null ? getOperationById(operationId, ifacePort) : null;
					if (operation!=null) {
						systemCall.setOperation(operation);
					} else {
						systemCall.setOperation(null);
						logger.debug("Cannot re-set operation reference for systemcall="+systemCall);
					}
				}
			}
			
			try {
				rUsageModel.save(null);
			} catch (IOException e) {
				throw new ResourceException(0, usageModel.getFullPath(), e.getMessage(), e);
			}			
		}
		

		private URI getModelURI(IFile file) {
			return URI.createPlatformResourceURI(file.getFullPath().toOSString(), false);			
		}
		
		private InterfacePort getInterfacePortById(String id, Resource rModel) {
			TreeIterator<EObject> it = EcoreUtil.getAllContents(rModel, true);
			while (it.hasNext()) {
				EObject eo = it.next();
				if (eo instanceof InterfacePort) {
					if (id.equals(((InterfacePort) eo).getId())) {
						return (InterfacePort) eo;
					}
				}
			}
			return null;
		}
		
		private Operation getOperationById(String id, InterfacePort ifacePort) {
			Collection<Operation> operations = ifacePort.getInterfaceType().getSignatures();
			Iterator<Operation> it = operations.iterator();
			while (it.hasNext()) {
				Operation op = it.next();				
				if (id.equals(op.getId())) {
					return (Operation) op;
				}				
			}
			return null;
		}
	}
}
