/**
 * 
 */
package eu.qimpress.ide.backbone.core.internal.model;

import java.lang.reflect.InvocationTargetException;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;

import eu.qimpress.ide.backbone.core.QImpressCore;
import eu.qimpress.ide.backbone.core.model.IQElement;
import eu.qimpress.ide.backbone.core.model.IQWorkspaceController;
import eu.qimpress.ide.backbone.core.model.InitializationException;

/**
 * Abstract controller for Q-ImPrESS application model entities. 
 *
 * @author Michal Malohlava
 *
 */
public abstract class AbstractQController<T extends IQElement> implements IQWorkspaceController {
	
	private static final Logger logger = Logger.getLogger(AbstractQController.class);
	
	protected T element;
	
	private boolean initializationInProgress = false;
	
	public AbstractQController(T element) {
		this.element = element;
	}

	@Override
	public synchronized Job init(boolean fork, boolean waitForFinish) throws InitializationException {
		initializationInProgress = true;		
		preInit();
		
		try {
			if (fork) {
				Job job = new InitializationJob();
				job.schedule();
				
				if (waitForFinish) {
					try {
						job.join();
					} catch (InterruptedException e) {					
					}				
				}
				
				return job;
			} else {
				try {
					doInit(new NullProgressMonitor());
				} catch (InterruptedException e) {				
					logger.error("Canceled initialization of element: " + element);				
				} catch (Exception e) {				
					logger.error("Cannot initialize element: " + element, e);				
					throw new InitializationException(e);
				} finally {				
				}
				
				return null;
			}
		} finally {
			initializationInProgress = false;
		}
	};
	
	protected abstract void doInit(IProgressMonitor monitor) throws CoreException, InvocationTargetException, InterruptedException;
	protected void preInit() {};

	public boolean isInitializationInProgress() {
		return initializationInProgress;
	}

	private class InitializationJob extends Job {
		
		private static final String JOB_NAME_PREFIX = "Initialization job of ";
		
		public InitializationJob() {
			super(JOB_NAME_PREFIX + element.getElementType());			
			setSystem(true);
			setPriority(SHORT);
		}

		@Override
		protected IStatus run(IProgressMonitor monitor) {
			try {
				monitor.beginTask("", 1);
				try {
					doInit(monitor);
				} catch (InterruptedException e) {					
					logger.warn("Cancel of element initialization: " + element);
					return new Status(IStatus.CANCEL, QImpressCore.PLUGIN_ID, "Canceled");					
				} catch (Exception e) {
					
					logger.error("Cannot initialize element: " + element, e);
					return new Status(IStatus.ERROR, QImpressCore.PLUGIN_ID, e.getMessage());
				}
			} finally {		
				initializationInProgress = false;
				monitor.done();				
			}
			
			return Status.OK_STATUS;
		}		
	}
}
