/**
 * 
 */
package eu.qimpress.ide.backbone.core.internal.listeners;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.runtime.CoreException;

import eu.qimpress.ide.backbone.core.QImpressCore;
import eu.qimpress.ide.backbone.core.QImpressNature;
import eu.qimpress.ide.backbone.core.model.IQApplicationModel;
import eu.qimpress.ide.backbone.core.model.IQInitializer;
import eu.qimpress.ide.backbone.core.model.IQProject;
import eu.qimpress.ide.backbone.core.model.IQRepository;
import eu.qimpress.ide.backbone.core.model.IQWorkspaceController;
import eu.qimpress.ide.backbone.core.model.QImpressApplicationModelManager;
import eu.qimpress.ide.backbone.core.model.RepositoryException;

/**
 * Q-ImPrESS model resource change listener for Q-I project closing and deleting
 * .
 * 
 * It listens to changes at Q-I projects and performs appropriate actions (i.e.,
 * deletion of project means closing alternatives database.)
 * 
 * @author Michal Malohlava
 * 
 */
public class QImpressProjectlResourceChangedListener implements
		IResourceChangeListener {

	private static final Logger logger = Logger
			.getLogger(QImpressProjectlResourceChangedListener.class);

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eclipse.core.resources.IResourceChangeListener#resourceChanged(org
	 * .eclipse.core.resources.IResourceChangeEvent)
	 */
	@Override
	public void resourceChanged(IResourceChangeEvent event) {		
		
		if (event.getType() == IResourceChangeEvent.PRE_DELETE
				|| event.getType() == IResourceChangeEvent.PRE_CLOSE) {

			IResource resource = event.getResource();
			logger.trace("PRE_DELETE event received on resource " + resource);
			if (resource != null && resource.getType() == IResource.PROJECT) {
				IProject project = (IProject) resource;

				handleProjectDelete(project);
			}

		} else if (event.getType() == IResourceChangeEvent.POST_CHANGE) {
			
			IResourceDeltaVisitor deltaVisitor = new IResourceDeltaVisitor() {				
				@Override
				public boolean visit(IResourceDelta delta) throws CoreException {					
					if (delta.getKind() == IResourceDelta.ADDED 
							|| 
						delta.getKind() == IResourceDelta.CHANGED && (delta.getFlags() & IResourceDelta.OPEN) != 0 ) {
						if (delta.getResource().getType() == IResource.PROJECT) {							
							IProject project = (IProject) delta.getResource();
							logger.trace("POST_CHANGE event received on project: " + project);
							
							if (QImpressNature.hasThisNature(project)) {								
								handleProjectAdded(project);								
							}
							
							return true;
						}
					}
					return true;
				}
			};
			
			try {
				event.getDelta().accept(deltaVisitor);				
			} catch (CoreException e) {				
				logger.warn("Error occurred during the resource delta visiting. ", e);
			}
		}
	}

	protected void handleProjectDelete(IProject project) {
		try {
			IQApplicationModel qAppModel = QImpressApplicationModelManager
					.getManager().getQAppModel();
			IQProject qProject = qAppModel.getQProject(project);

			if (qProject != null) {
				logger.trace("PRE_DELETE event performed on Q-I project "
						+ qProject);

				// close repository and remove project
				qAppModel.removeQProject(project);

				// TODO FEATURE check if we need somehow to unload
				// ShadowModelEditor

			}
		} catch (RepositoryException e) {
			logger.error("Cannot close repository object for Q-I project: "
					+ project, e);
		}
	}

	protected void handleProjectAdded(IProject project) {
		try {
			IQProject qProject = QImpressCore.getQProject(project);
			IQRepository qRepository = qProject.getRepository();
			IQInitializer initilizer = (IQInitializer) qRepository.getAdapter(IQInitializer.class);
			
			if (initilizer.needsInitialization()) {
				IQWorkspaceController controller = (IQWorkspaceController) qRepository.getAdapter(IQWorkspaceController.class);
				controller.init(true, false);
			}
		} catch (RepositoryException e) {
			logger.error("Cannot handle project addition.", e);
		}
	}
}
