/**	
 * 
 */
package eu.qimpress.ide.backbone.core.internal.listeners;


import java.util.ArrayList;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.runtime.CoreException;

import eu.qimpress.ide.backbone.core.QImpressCore;
import eu.qimpress.ide.backbone.core.QImpressNature;
import eu.qimpress.ide.backbone.core.model.IQModel;
import eu.qimpress.ide.backbone.core.model.QImpressApplicationModelManager;

/**
 * Resource change listener.
 * 
 * It propagates information about underlying resource change through the backbone infrastructure.
 *  
 * @author Michal Malohlava
 *
 */
public class ModelResourceChangedListener implements IResourceChangeListener {
	
	private static final Logger logger = Logger.getLogger(ModelResourceChangedListener.class);

	/* (non-Javadoc)
	 * @see org.eclipse.core.resources.IResourceChangeListener#resourceChanged(org.eclipse.core.resources.IResourceChangeEvent)
	 */
	@Override
	public void resourceChanged(IResourceChangeEvent event) {
		// we are interested only in change events	
		if (event.getType() == IResourceChangeEvent.POST_CHANGE) {
			IResourceDelta resourceDelta = event.getDelta();			
			
			handleChangedResource(resourceDelta);
		}
	}

	private void handleChangedResource(IResourceDelta resourceDelta) {
		final ArrayList<IQModel> changedModels = new ArrayList<IQModel>();
		final ArrayList<IQModel> addedModels = new ArrayList<IQModel>();
		final ArrayList<IQModel> deletedModels = new ArrayList<IQModel>();
        
		IResourceDeltaVisitor visitor = new IResourceDeltaVisitor() {
           public boolean visit(IResourceDelta delta) {
        	   
              if (delta.getKind() == IResourceDelta.CHANGED) {                 
	              //only interested in content changes
	              if ((delta.getFlags() & IResourceDelta.CONTENT) == 0)
	                 return true;
	            
	              IResource resource = delta.getResource();
	              //only interested in model files
	              IQModel model = getQModel(resource);
	              if (model!=null) {
	            		 logger.trace("Model changed: " + model);
	            		 changedModels.add(model);
            	  }	              
              } else if (delta.getKind() == IResourceDelta.ADDED) {
            	  IResource resource = delta.getResource();
            	  
            	  IQModel model = getQModel(resource);
            	  if (model!=null) {
        			  logger.trace("Model added: " + model);
        			  addedModels.add(model);
        		  }
              } else if (delta.getKind() == IResourceDelta.REMOVED) {
            	  IResource resource = delta.getResource();
            	  IQModel model = getQModel(resource);
            	  if (model!=null) {
        			  logger.trace("Model deleted: " + model);
        			  deletedModels.add(model);
        		  }           	  
              } 
              
              return true;
           }
        };
        
        try {        	
        	resourceDelta.accept(visitor);        	
        	
        	// TODO Question? To decrease overhead - create a separate job informing about new and modified models or do it in the same thread?
        	
        	// process changed models
        	if (changedModels.size() > 0)
        		QImpressApplicationModelManager.getManager().getQAppModel().fireModelModified(changedModels.toArray(new IQModel[changedModels.size()]));
        	        	
        	// process added models
        	if (addedModels.size() > 0)
        		QImpressApplicationModelManager.getManager().getQAppModel().fireModelCreated(addedModels.toArray(new IQModel[addedModels.size()]));
        	
        	// process deleted models
        	if (deletedModels.size() > 0)
        		QImpressApplicationModelManager.getManager().getQAppModel().fireModelDeleted(deletedModels.toArray(new IQModel[deletedModels.size()]));        	

        	        	
        } catch (CoreException e) {
        	logger.error("Cannot proceed changes in the workspace because of " + e.getMessage(), e);
        }		
	}
	
	private IQModel getQModel(IResource resource) {
		if (resource.getType() == IResource.FILE && QImpressNature.hasThisNature(resource.getProject())) {
  		  IQModel model = QImpressCore.getModelByFile((IFile) resource);
  		  return model;
  	  	}
		
		return null;		
	}

}
