/**
 * 
 */
package eu.qimpress.ide.backbone.core.internal.listeners;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.ui.actions.WorkspaceModifyOperation;

import eu.qimpress.ide.backbone.core.QImpressCore;
import eu.qimpress.ide.backbone.core.internal.model.QCachedDirectoryRepositoryImpl;
import eu.qimpress.ide.backbone.core.model.IQAlternative;
import eu.qimpress.ide.backbone.core.model.IQProject;
import eu.qimpress.ide.backbone.core.model.IQRepository;
import eu.qimpress.ide.backbone.core.model.RepositoryException;

/**
 * Resource listener handling delete of alternative by external user.
 *
 * @author Michal Malohlava
 *
 */
public class AlternativeResourceChangedListener implements IResourceChangeListener {
	
	private static final Logger logger = Logger.getLogger(AlternativeResourceChangedListener.class);

	@Override
	public void resourceChanged(IResourceChangeEvent event) {				
		// we are interested only in change events	
		if (event.getType() == IResourceChangeEvent.POST_CHANGE) {
			IResourceDelta resourceDelta = event.getDelta();			
			
			handleChangedResource(resourceDelta);
		}
	}

	private void handleChangedResource(IResourceDelta resourceDelta) {
		final ArrayList<IQAlternative> deletedAlternatives = new ArrayList<IQAlternative>();
		
		IResourceDeltaVisitor deltaVisitor = new IResourceDeltaVisitor() {
			
			@Override
			public boolean visit(IResourceDelta delta) throws CoreException {
				if (delta.getKind() == IResourceDelta.REMOVED) {
					if (delta.getResource().getType() == IResource.FOLDER) {
						String alternativeId = getAlternativeId((IFolder) delta.getResource());
						if (alternativeId != null) {
							try {
								IProject project = delta.getResource().getProject();
								IQProject qProject = QImpressCore.getQProject(project);
								if (qProject != null) { 
									IQAlternative qAlternative = qProject.getRepository().getAlternative(alternativeId);
									if (qAlternative != null) {
										logger.trace("Alternative " + qAlternative + " was added to a list of deleted alternatives.");
										deletedAlternatives.add(qAlternative);
									}
								} else {
									logger.warn("The Q-I project does not exist for the Eclipse project: " + project);
								}
							} catch (RepositoryException e) {
								logger.error("Some error occured during getting an alternative. Skipped.", e);
							}
						}
					}
				}				
				return true;
			}
		};
		
		try {
			resourceDelta.accept(deltaVisitor);
			
			processDeletedAlternatives(deletedAlternatives);
		} catch (CoreException e) {
			logger.warn("Error occured during processsing deleted alternative", e);
		}		
	}
	
	private void processDeletedAlternatives(final ArrayList<IQAlternative> deletedAlternatives) {
		Job job = new ClearAlternativesJob(deletedAlternatives);
		job.schedule();
	}

	private String getAlternativeId(IFolder folder) {
		// [project]/alternatives/01232-2-321-312-31/
		String repositoryName = folder.getFullPath().segment(1);
		// check if deleted folder was an alternative's folder
		if (QCachedDirectoryRepositoryImpl.DEFAULT_REPOSITORY_LOCATION.equals(repositoryName)) {
			return folder.getFullPath().lastSegment();									
		}
		
		return null;
	}
	
	private static final class ClearAlternativesJob extends Job {
		
		private List<IQAlternative> deletedAlternatives;

		public ClearAlternativesJob(List<IQAlternative> deletedAlternatives) {
			super("Clear alternatives");
			setSystem(true);
			setPriority(SHORT);			
			
			this.deletedAlternatives = deletedAlternatives;
		}

		@Override
		protected IStatus run(IProgressMonitor monitor) {
			// do the operation in WS operation locking whole workspace
			WorkspaceModifyOperation operation = new WorkspaceModifyOperation() {
				
				@Override
				protected void execute(IProgressMonitor monitor) throws CoreException,
						InvocationTargetException, InterruptedException {
					
					monitor.beginTask("", deletedAlternatives.size());
					try {
						for (IQAlternative alt : deletedAlternatives) {
							IQRepository qRepository = alt.getRepository();
							logger.trace("Clearing alternative, because its folder does not exist. Alternative: " + alt);
							try {
								qRepository.deleteAlternative(alt, false, true);
							} catch (RepositoryException e) {							
								logger.warn("Cannot clear alternative: " + alt, e);
							}
						}
					} finally {
						monitor.done();
					}
				}
			};
			
			monitor.beginTask("", deletedAlternatives.size());
			try {
				operation.run(new SubProgressMonitor(monitor, deletedAlternatives.size()));
			} catch (Exception e) {
				logger.error("Exception occured during clearing alternatives!", e);
			} finally {
				monitor.done();
			}
			
			return Status.OK_STATUS ;
		}		
	}
}
