package eu.qimpress.ide.backbone.core;


import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;

import eu.qimpress.ide.backbone.core.internal.listeners.AlternativeResourceChangedListener;
import eu.qimpress.ide.backbone.core.internal.listeners.ModelResourceChangedListener;
import eu.qimpress.ide.backbone.core.internal.listeners.QImpressProjectlResourceChangedListener;
import eu.qimpress.ide.backbone.core.model.IQAlternative;
import eu.qimpress.ide.backbone.core.model.IQApplicationModel;
import eu.qimpress.ide.backbone.core.model.IQElement;
import eu.qimpress.ide.backbone.core.model.IQModel;
import eu.qimpress.ide.backbone.core.model.IQProject;
import eu.qimpress.ide.backbone.core.model.IQRepository;
import eu.qimpress.ide.backbone.core.model.QImpressApplicationModelManager;
import eu.qimpress.ide.backbone.core.model.RepositoryException;
import eu.qimpress.ide.backbone.core.model.IQElement.ElementType;

/**
 * The activator class controls the plug-in life cycle
 */
public class QImpressCore extends AbstractUIPlugin {

	// The plug-in ID
	public static final String PLUGIN_ID = "eu.qimpress.ide.backbone.core";
	
	// The shared instance
	private static QImpressCore plugin;
	
	// workspace listeners
	private ModelResourceChangedListener modelResourceChangedListener = new ModelResourceChangedListener();
	private QImpressProjectlResourceChangedListener qImpressProjectlResourceChangedListener = new QImpressProjectlResourceChangedListener();
	private AlternativeResourceChangedListener alternativeResourceChangedListener = new AlternativeResourceChangedListener();
	
	/**
	 * The constructor
	 */
	public QImpressCore() {
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.ui.plugin.AbstractUIPlugin#start(org.osgi.framework.BundleContext)
	 */
	public void start(BundleContext context) throws Exception {
		// start plugin
		super.start(context);
		plugin = this;
		
		// register workspace listeners		
		IWorkspace workspace = ResourcesPlugin.getWorkspace();
		workspace.addResourceChangeListener(this.modelResourceChangedListener);
		workspace.addResourceChangeListener(this.qImpressProjectlResourceChangedListener, IResourceChangeEvent.PRE_DELETE | IResourceChangeEvent.PRE_CLOSE | IResourceChangeEvent.POST_CHANGE);
		workspace.addResourceChangeListener(this.alternativeResourceChangedListener, IResourceChangeEvent.POST_CHANGE);
		
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.ui.plugin.AbstractUIPlugin#stop(org.osgi.framework.BundleContext)
	 */
	public void stop(BundleContext context) throws Exception {
		// deregister workspace listeners
		IWorkspace workspace = ResourcesPlugin.getWorkspace();
		workspace.removeResourceChangeListener(this.modelResourceChangedListener);
		workspace.removeResourceChangeListener(this.qImpressProjectlResourceChangedListener);
		workspace.removeResourceChangeListener(this.alternativeResourceChangedListener);

		// stop plugin
		plugin = null;
		super.stop(context);
	}

	/**
	 * Returns the shared instance
	 *
	 * @return the shared instance
	 */
	public static QImpressCore getDefault() {
		return plugin;
	}
	
	/* ====== helper factory methods ====== */
	public static IQElement create(IResource resource) {
		return QImpressApplicationModelManager.create(resource, null);
	}
	
	public static IQElement create(IResource resource, IQProject project) {
		return QImpressApplicationModelManager.create(resource, project);
	}
	
	public static IQApplicationModel create(IWorkspaceRoot root) {
		if (root == null) {
			return null;
		}
		
		return QImpressApplicationModelManager.getManager().getQAppModel();
	}
	
	public static IQProject create(IProject project) {
		if (project == null) {
			return null;
		}
	
		IQApplicationModel qAppModel = QImpressApplicationModelManager.getManager().getQAppModel();
		
		return qAppModel.getQProject(project);
	}
	
	public static IQElement create(IFolder folder) {
		return QImpressApplicationModelManager.create(folder, null);
	}
	
	public static IQElement create(IFile file) {
		return QImpressApplicationModelManager.create(file, null);
	}
	
	/**
	 * Return an alternative with given id.
	 * 
	 * WARNING! This method shouldn't be called for predefined alternative id - e.g. '0' (common models)
	 * 
	 * @param id unique id of alternative
	 * @return
	 */
	public static IQAlternative getAlternativeById(String id) {
		
		IQElement element = QImpressApplicationModelManager.getManager().getQAppModel().getElementByID(id);
		
		if (element != null && element.getElementType() == ElementType.Q_ALTERNATIVE) {
			return (IQAlternative) element;			
		}
		
		return null;
	}
	
	public static IQModel getModelByFile(IFile file) {
		return QImpressApplicationModelManager.getModelByResource(file);
	}
	
	/**
	 * Return project containing a resource referenced by given URI
	 * 
	 * @param platformURI - URI has to be a platform-URI (i.e., 'platform://...') or a file://
	 * @return project containing given resource or null
	 */
	public static IProject getProject(URI platformURI) {
		IProject result = null;
		
		if (platformURI.isPlatform()) {
			String projectName = URI.decode(platformURI.segment(1)); // URI is in the format platform://resource/<project_name>/<path_to_resource>
			IWorkspaceRoot wRoot = ResourcesPlugin.getWorkspace().getRoot();
			
			result = wRoot.getProject(projectName);
		} else if (platformURI.isFile()) {			
			IPath filePath = new Path(platformURI.toFileString());
			IPath workspacePath = ResourcesPlugin.getWorkspace().getRoot().getLocation();
			filePath = filePath.makeRelativeTo(workspacePath);
			
			String projectName = filePath.segment(0);
			result = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
		}
		
		return result;
	}
	
	public static String getAlternativeId(URI resourceURI) {
		String result = null;
		
		if (resourceURI.isPlatform()) {
			result = URI.decode(resourceURI.segment(3)); // URI is in the format platform://resource/<project_name>/alternatives/<alternative-d>
		} else if (resourceURI.isFile()) {
			IPath filePath = new Path(resourceURI.toFileString());
			IPath workspacePath = ResourcesPlugin.getWorkspace().getRoot().getLocation();
			filePath = filePath.makeRelativeTo(workspacePath);
			
			result = filePath.segment(2);
		}
		
		return result;		
	}
	
	/**
	 * Derives aternativeId from filesystem path.
	 * @param resourceLocation Path to the resource
	 * @return alternativeId
	 * @author Viliam Simko
	 */
	public static String getAlternativeId(IPath resourceLocation) {
		IPath workspacePath = ResourcesPlugin.getWorkspace().getRoot().getLocation();
		resourceLocation = resourceLocation.makeRelativeTo(workspacePath);
		return resourceLocation.segment(2);
	}
	
	/**
	 * Transforms reference from IProject to IQProject
	 * @param project IProject
	 * @return IQProject
	 */
	public static IQProject getQProject(IProject project) {
		return QImpressApplicationModelManager.getManager().getQAppModel().getQProject(project);
	}

	/**
	 * Helper method that simplifies access to the Q-Impress project for a given resource referenced by URI
	 * 
	 * @param platformURI URI has to be a platform-URI (i.e., 'platform://...') or a file://
	 * @return Q-Impress project containing the given resource or null
	 * 
	 * @see QImpressCore#getProject(URI)
	 * @see QImpressCore#getQProject(IProject)
	 * @author Viliam Simko
	 */
	public static IQProject getQProject(URI platformURI)
	{
		IProject project = QImpressCore.getProject( platformURI );
		return QImpressCore.getQProject(project);
	}
	
	/**
	 * Helper method returns Q-Impress project by name.
	 * @author Viliam Simko
	 */
	public static IQProject getQProject(String projectName)
	{
		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(projectName);
		return getQProject(project);
	}
	
	/**
	 * Helper method that simplifies access to the Q-Impress project for a given EObject.
	 * 
	 * @param eObject
	 * @return
	 * 
	 * @see QImpressCore#getQProject(URI)
	 * @author Viliam Simko
	 */
	public static IQProject getQProject(EObject eObject)
	{
		return getQProject(eObject.eResource().getURI());
	}

	/**
	 * Helper method that can identify global alternatives.
	 * At the moment we identify global alternative by description.
	 * 
	 * @return TRUE if the alternative is global
	 * 
	 * @see IQRepository#GLOBAL_ALTERNATIVE_NAME
	 * @see IQRepository#getGlobalAlternative()
	 * @author Viliam Simko 
	 */
	public static boolean isGlobalAlternative(IQAlternative alternative)
	{
		String alternativeID = alternative.getInfo().getId();
		return IQRepository.GLOBAL_ALTERNATIVE_ID.equals( alternativeID );
	}
	
	/**
	 * Helper method. 
	 * @param eObject
	 * @return IQModel where the given EObject is stored.
	 * @throws RepositoryException
	 * @author Viliam Simko
	 */
	public static IQModel getResultModelByEObject(EObject eObject) throws RepositoryException {
		IQRepository repo = QImpressCore.getQProject( eObject ).getRepository();
		IQModel model = repo.getResultModel();
		return model;
	}

}
