/**
 * 
 */
package eu.qimpress.ide.backbone.core.ui.tabs;

import org.apache.log4j.Logger;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;

import eu.qimpress.ide.backbone.core.QImpressCore;
import eu.qimpress.ide.backbone.core.model.IQAlternative;
import eu.qimpress.ide.backbone.core.model.IQProject;
import eu.qimpress.ide.backbone.core.model.QImpressApplicationModelManager;
import eu.qimpress.ide.backbone.core.model.RepositoryException;
import eu.qimpress.ide.backbone.core.ui.widgets.UsageModelViewerFactory;
import eu.qimpress.ide.backbone.core.utils.SAMMModelsUtils;
import eu.qimpress.ide.backbone.core.utils.SAMMModelsUtils.EnumModelLookup;
import eu.qimpress.samm.usagemodel.UsageModel;

/**
 * Launch configuration tab for selecting usage models.
 * 
 * It allows enforcing the alternative, then it show a usage models in the selected alternative.
 * 
 * @author Michal Malohlava
 *
 */
public class QImpressUsageModelSelectionTab extends AbstractLaunchConfigurationTab implements IAlternativeSelection {
	public static final String SELECTED_USAGE_MODEL_ID = "eu.qimpress.ide.backbone.core.ui.UsageModelSelection.id";
	public static final String SELECTED_USAGE_MODEL_PROJECT_NAME = "eu.qimpress.ide.backbone.core.ui.UsageModelSelection.project";
	public static final String SELECTED_USAGE_MODEL_ALTERNATIVE_ID = "eu.qimpress.ide.backbone.core.ui.UsageModelSelection.alternative.id";
	
	private static final Logger logger = Logger.getLogger(QImpressUsageModelSelectionTab.class);
	
	private static final boolean DEFAULT_VIEW_MODE_FLAT = false; 
	
	private CheckboxTreeViewer usageModelViewer;
	private Label selectionText;
	private boolean usageModelsFlatMode = DEFAULT_VIEW_MODE_FLAT;
	
	private IQAlternative selectedAlternative;

	@Override
	public void createControl(Composite parent) {
		Composite myTabControl = new Composite(parent,SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		gridLayout.marginWidth = 5;
		gridLayout.marginHeight = 5;
		gridLayout.numColumns = 2;
		myTabControl.setLayout(gridLayout);
		setControl(myTabControl);
		
		Group grpSelectUsageModel = new Group(myTabControl, SWT.FILL);
		grpSelectUsageModel.setText("Select usage model");
		GridLayout selectorLayout = new GridLayout();
		selectorLayout.marginWidth = 5;
		selectorLayout.marginHeight = 5;
		selectorLayout.verticalSpacing = 0;
		selectorLayout.numColumns = 1;
		grpSelectUsageModel.setLayout(selectorLayout);
		grpSelectUsageModel.setLayoutData(new GridData(SWT.FILL,SWT.FILL,true,true,2,1));
			
		usageModelViewer = (CheckboxTreeViewer) UsageModelViewerFactory.createWidgetFactory(grpSelectUsageModel).getWidget();
		usageModelViewer.getControl().setLayoutData(new GridData(SWT.FILL,SWT.FILL,true,true,1,1));
		usageModelViewer.expandAll();
		usageModelViewer.addCheckStateListener(new ICheckStateListener() {
			
			/**
			 * Called when the user manually click the checkbox.
			 */
			public void checkStateChanged(CheckStateChangedEvent event) {
				setDirty(true);
				updateLaunchConfigurationDialog();
				updateCheckedItemLabelText();
			}
		});
		
		/* button for showing all alternatives */
		final Button filterSelectionButton = new Button(grpSelectUsageModel, SWT.TOGGLE | SWT.FLAT);
		filterSelectionButton.setText("Show all");
		filterSelectionButton.setLayoutData(new GridData(SWT.RIGHT,SWT.BOTTOM,false,false,1,1));
		filterSelectionButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				enableFiltering(!filterSelectionButton.getSelection());
			}
		});
		
		Label selection = new Label(myTabControl,SWT.NONE);
		selection.setText("Selected usage model: ");
		selection.setLayoutData(new GridData(SWT.FILL,SWT.FILL,false,false,1,1));
		
		selectionText = new Label(myTabControl,SWT.NONE);
		selectionText.setText("<none>");
		selectionText.setLayoutData(new GridData(SWT.FILL,SWT.FILL,false,false,1,1));		
	}

	protected void updateCheckedItemLabelText() {
		Object[] checkedUsageModel = usageModelViewer.getCheckedElements();
		
		if (checkedUsageModel.length > 0 && checkedUsageModel[0] instanceof UsageModel) {
			selectionText.setText( "'" + ((UsageModel) checkedUsageModel[0]).getName() + "' in '" + this.selectedAlternative.getInfo().getDescription() + "'");			
		} else {
			selectionText.setText("<none>");
		}
	}

	@Override
	public String getName() {		
		return "Q-ImPrESS Usage Model";
	}

	@Override
	public void initializeFrom(ILaunchConfiguration configuration) {	
		
		String cfgUsageModelId = null;
		String cfgAlternativeId = null;
		String cfgProjectName = null;		
		
		try {
			cfgUsageModelId = configuration.getAttribute(SELECTED_USAGE_MODEL_ID, (String) null);
			cfgAlternativeId = configuration.getAttribute(SELECTED_USAGE_MODEL_ALTERNATIVE_ID, (String) null);
			cfgProjectName = configuration.getAttribute(SELECTED_USAGE_MODEL_PROJECT_NAME, (String) null);
			
			if (cfgUsageModelId != null && cfgAlternativeId != null && cfgProjectName != null) {
				if (selectedAlternative != null) {
					if (
						(selectedAlternative.getInfo().getId().equals(cfgAlternativeId) && this.usageModelsFlatMode)
							||
						!this.usageModelsFlatMode
						) {
						setUsageModelInView(this.selectedAlternative, cfgUsageModelId);
					}
				} else if (selectedAlternative == null && !this.usageModelsFlatMode) {
					initializeFormInternal(cfgUsageModelId, cfgAlternativeId, cfgProjectName);
				}
			}
		} catch (Exception e) {
			logger.warn("Cannot initialize usage model selection form", e);
		}		
	}
	
	protected void initializeFormInternal(String usageModelId, String alternativeId, String projectName) throws RepositoryException {
		IQProject project = QImpressCore.getQProject(projectName);		
		if (project == null) return;
		IQAlternative alternative = project.getRepository().getAlternative(alternativeId);
		if (alternative == null) return;
		
		setUsageModelInView(alternative, usageModelId);
	}
	
	protected void setUsageModelInView(IQAlternative alternative, String usageModelId) throws RepositoryException {
		EnumModelLookup lookup = UsageModelViewerFactory.getModelLookup(this.usageModelViewer);
		UsageModel[] usageModels = SAMMModelsUtils.getUsageModels(alternative, lookup);
				
		for (UsageModel usageModel : usageModels) {
			if (usageModel.getId().equals(usageModelId)) {
				this.setSelectedAlternative(alternative);
				this.usageModelViewer.setChecked(usageModel, true);
				updateCheckedItemLabelText();
				break;
			}								
		}
	}

	@Override
	public void performApply(ILaunchConfigurationWorkingCopy configuration) {
		Object[] elements = usageModelViewer.getCheckedElements();
		if (elements.length != 1 || !(elements[0] instanceof UsageModel)) {
			configuration.setAttribute(SELECTED_USAGE_MODEL_ID, "");
		} else {
			UsageModel selectedUsageModel = (UsageModel) elements[0];
			
			// we need this value to identify selected (checked) item
			configuration.setAttribute(SELECTED_USAGE_MODEL_ID, selectedUsageModel.getId());
			configuration.setAttribute(SELECTED_USAGE_MODEL_ALTERNATIVE_ID, QImpressCore.getAlternativeId(selectedUsageModel.eResource().getURI()));									
			// we need this value to get the repository
			String projectName = QImpressCore.getQProject(selectedUsageModel).getProject().getName();
			configuration.setAttribute(SELECTED_USAGE_MODEL_PROJECT_NAME, projectName);
		}		
	}

	@Override
	public void setDefaults(ILaunchConfigurationWorkingCopy configuration) {
		configuration.setAttribute(SELECTED_USAGE_MODEL_ID, (String) null);
		configuration.setAttribute(SELECTED_USAGE_MODEL_ALTERNATIVE_ID, (String) null);
		configuration.setAttribute(SELECTED_USAGE_MODEL_PROJECT_NAME, (String) null);		
	}
	
	@Override
	public boolean isValid(ILaunchConfiguration launchConfig) {
		setErrorMessage(null);
		
		return validatePage();		
	}

	protected boolean validatePage() {
		Object[] selectedObject = this.usageModelViewer.getCheckedElements();
		if (selectedObject.length != 1) {
			setErrorMessage("Usage model has to be selected");
			
			return false;
		}
		
		return true;
	}

	/**
	 * Enforce selected alternative.
	 * 
	 * @param alternative 
	 */
	public void setSelectedAlternative(IQAlternative alternative) {
		if (alternative != this.selectedAlternative) {
			this.selectedAlternative = alternative;
			
			// first we need to setup input
			if (usageModelsFlatMode) {
				this.usageModelViewer.setInput(alternative);
			} else {
				this.usageModelViewer.setInput(QImpressApplicationModelManager.getManager().getQAppModel());
			}
			
			// and then we can filter
			enableFiltering(selectedAlternative != null);
			
			usageModelViewer.expandAll();		
			setDirty(true);
		}
	}
	
	/**
	 * Set flat mode to view a list of usage models. If false is specified then also the parent alternative
	 * and project are shown.
	 * 
	 * @param mode true for flat mode, false for hierarchical model, flat is default
	 */
	public void setFlatViewMode(boolean mode) {
		this.usageModelsFlatMode = mode;
		
		UsageModelViewerFactory.setFlatViewMode(mode, this.usageModelViewer);		
	}
	
	/**
	 * Return selected UsageModel or null
	 * 
	 * @return selected instance of UsageModel or null if no model is selected.
	 */
	public UsageModel getSelectedUsageModel() {
		Object[] selectedObject = this.usageModelViewer.getCheckedElements();
		if (selectedObject.length == 1) {
			return (UsageModel) selectedObject[0];
		} else {
			return null;
		}		
	}
	
	protected void enableFiltering(boolean enable) {
		// and then we can filter		
		if (enable && selectedAlternative != null) {
			UsageModelViewerFactory.applyFilterFor(selectedAlternative, this.usageModelViewer);
		} else {
			usageModelViewer.resetFilters();
		}		
	}
}
