/**
 * 
 */
package eu.qimpress.ide.backbone.core.ui.tabs;

import org.apache.log4j.Logger;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyAdapter;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import eu.qimpress.ide.backbone.core.QImpressCore;
import eu.qimpress.ide.backbone.core.model.IQAlternative;
import eu.qimpress.ide.backbone.core.model.IQModel;
import eu.qimpress.ide.backbone.core.model.IQProject;
import eu.qimpress.ide.backbone.core.model.IQRepository;
import eu.qimpress.ide.backbone.core.model.RepositoryException;
import eu.qimpress.ide.backbone.core.ui.widgets.QAlternativeEvaluationSelectorWidgetFactory;
import eu.qimpress.ide.backbone.core.ui.widgets.QAlternativesCheckBoxTreeViewer;
import eu.qimpress.ide.backbone.core.ui.widgets.QAlternativesTreeViewerFactory;
import eu.qimpress.ide.backbone.core.ui.widgets.QAlternativesTreeViewerFactory.SelectAlternativesEnum;
import eu.qimpress.resultmodel.AlternativeEvaluation;

/**
 * Tab for selecting alternative evaluations.
 *  
 * @author Michal Malohlava
 * @author Viliam Simko
 */
public class QImpressAlternativeEvaluationSelectionTab 
extends	AbstractLaunchConfigurationTab implements IAlternativeSelection {
	
	public static final String SELECTED_ALTERNATIVE_EVALUATION_ID = "eu.qimpress.ide.backbone.core.ui.AlternativeEvaluationSelection.id";
	public static final String SELECTED_PROJECT_NAME = "eu.qimpress.ide.backbone.core.ui.AlternativeEvaluationSelection.project";

	private static final Logger logger = Logger.getLogger(AbstractLaunchConfigurationTab.class);

	private CheckboxTreeViewer alternativeEvaluationViewer;
	private Label selectionText;
	private IQAlternative selectedAlternative;
	
	@Override
	public void createControl(Composite parent) {
		Composite myTabControl = new Composite(parent,SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		gridLayout.marginWidth = 5;
		gridLayout.marginHeight = 5;
		gridLayout.numColumns = 2;
		myTabControl.setLayout(gridLayout);
		setControl(myTabControl);
		
		Group grpSelectQimpressAlternative = new Group(myTabControl, SWT.FILL);
		grpSelectQimpressAlternative.setText("Select Alternative Evaluation");
		GridLayout selectorLayout = new GridLayout();
		selectorLayout.marginWidth = 5;
		selectorLayout.marginHeight = 5;
		selectorLayout.verticalSpacing = 0;
		selectorLayout.numColumns = 1;
		grpSelectQimpressAlternative.setLayout(selectorLayout);
		grpSelectQimpressAlternative.setLayoutData(new GridData(SWT.FILL,SWT.FILL,true,true,2,1));
			
		alternativeEvaluationViewer = (CheckboxTreeViewer) QAlternativeEvaluationSelectorWidgetFactory.createWidgetFactory(grpSelectQimpressAlternative).getWidget();
		alternativeEvaluationViewer.getControl().setLayoutData(new GridData(SWT.FILL,SWT.FILL,true,true,1,1));		
		alternativeEvaluationViewer.expandAll();
		alternativeEvaluationViewer.addCheckStateListener(new ICheckStateListener() {
			
			/**
			 * Called when the user manually click the checkbox.
			 */
			public void checkStateChanged(CheckStateChangedEvent event) {
				setDirty(true);
				updateLaunchConfigurationDialog();
				updateCheckedItemLabelText();
			}
		});
		/* button for showing all alternatives */
		final Button filterSelectionButton = new Button(grpSelectQimpressAlternative, SWT.TOGGLE | SWT.FLAT);
		filterSelectionButton.setText("Show all");
		filterSelectionButton.setLayoutData(new GridData(SWT.RIGHT,SWT.BOTTOM,false,false,1,1));
		filterSelectionButton.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				enableFiltering(!filterSelectionButton.getSelection());
			}
		});
		
		Label selection = new Label(myTabControl,SWT.NONE);
		selection.setText("Selected Alternative Evaluation: ");
		selection.setLayoutData(new GridData(SWT.FILL,SWT.FILL,false,false,1,1));
		
		selectionText = new Label(myTabControl,SWT.NONE);
		selectionText.setText("<none>");
		selectionText.setLayoutData(new GridData(SWT.FILL,SWT.FILL,false,false,1,1));
				
		// creates additional buttons
		createNewAlternativeEvaluationButton(myTabControl);
		createDeleteSelectedAlternativeEvaluationButton(myTabControl);
	}
		
	/**
	 * A button that displays a dialog to add a new alternative evaluation. 
	 * @param parent Where the button should be installed
	 */
	private void createNewAlternativeEvaluationButton(final Composite parent)
	{
		Button createButton = new Button(parent, SWT.PUSH);
		createButton.setText("Create Alternative Evaluation");
		createButton.addListener(SWT.Selection, new Listener() {
			
			private QAlternativesCheckBoxTreeViewer treeViewer;
			private Text evalNameInputField;
			
			/**
			 * The input field should be disabled if no alternative is selected in the tree.
			 */
			private void updateInputFieldBehaviour()
			{
				// widgets must be initialized
				assert evalNameInputField != null;
				assert treeViewer != null;
				
				evalNameInputField.setEnabled( treeViewer.getFirstSelectedAlternative() != null );
			}
			
			@Override
			public void handleEvent(Event event) {
				
				// clicking the CREATE button should pop-up a dialog window 
				final Shell dialog = new Shell(SWT.DIALOG_TRIM | SWT.APPLICATION_MODAL);		    
			    dialog.setText("Select Alternative");
			    
			    // containing a grid layout with one column
			    // - first row is the input field
			    // - second row is the tree viewer
			    GridLayout gridLayout = new GridLayout();
			    gridLayout.marginWidth = 5;
			    gridLayout.marginHeight = 5;
			    dialog.setLayout(gridLayout);
			    
				// the group for input text and OK button
				Group inputFieldGroup = new Group(dialog, SWT.SHADOW_NONE);
				inputFieldGroup.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));
				inputFieldGroup.setText("Name of an alternative evaluation:");
				inputFieldGroup.setLayout(new GridLayout(3, false));
				
				// input field representing the name of a new alternative evaluation
				evalNameInputField = new Text(inputFieldGroup, SWT.SINGLE | SWT.BORDER);
				evalNameInputField.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

				// OK button
				final Button addButton = new Button(inputFieldGroup, SWT.PUSH);
				addButton.setLayoutData(new GridData(SWT.END, SWT.FILL, false, true));
				addButton.setText("Add");

				// CLOSE button
				final Button closeButton = new Button(inputFieldGroup, SWT.PUSH);
				closeButton.setLayoutData(new GridData(SWT.END, SWT.FILL, false, true));
				closeButton.setText("Close");

				// installs the tree viewer into the dialog window by using the factory
			    treeViewer =
					(QAlternativesCheckBoxTreeViewer) QAlternativesTreeViewerFactory
						.createTreeViewer(dialog, SelectAlternativesEnum.SELECT_ONE)
						.setCanSelectQRepository(false)
						.getTreeViewer();

			    // the tree fills the whole window
				treeViewer.getControl().setLayoutData( new GridData(SWT.FILL, SWT.FILL, true, true));

				// show expanded tree (may cause some problems with huge models)
				treeViewer.expandAll();

				// when selected, the check-box should automatically transfer focus to the input field
				treeViewer.addCheckStateListener(new ICheckStateListener() {
					
					@Override
					public void checkStateChanged(CheckStateChangedEvent event) {
						updateInputFieldBehaviour();
						evalNameInputField.forceFocus();
						evalNameInputField.selectAll();
					}
				});

				// initially disable the input field until the alternative is selected 
				updateInputFieldBehaviour();
				
				// installs the input field listener waiting for user to press ENTER key 
				evalNameInputField.addKeyListener(new KeyAdapter() {
					public void keyPressed(KeyEvent e) {
						if(e.character == SWT.LF || e.character == SWT.CR) {
							addButton.notifyListeners(SWT.Selection, new Event());
						} else if (e.character == SWT.ESC) {
							closeButton.notifyListeners(SWT.Selection, new Event());
						}
					};
				});
				
				// listener for OK button
				addButton.addSelectionListener(new SelectionAdapter() {
					public void widgetSelected(SelectionEvent e) {
						
						IQAlternative selectedAlternative = treeViewer.getFirstSelectedAlternative();
						
						// there should always be a single alternative selected in the tree
						assert selectedAlternative != null;
						
						try {
							// use the selected alternative to get the reference to the repository
							IQRepository repo = selectedAlternative.getRepository();
							
							// create a new object in the repository (empty alternative evaluation)
							AlternativeEvaluation newAlternativeEvaluation = repo.createAlternativeEvaluation();
							
							// fill some values to the empty alternative evaluation
							newAlternativeEvaluation.setName(evalNameInputField.getText());
							newAlternativeEvaluation.setAlternativeId( selectedAlternative.getInfo().getId() );
							
							// the alternative should be saved later
							IQModel model = QImpressCore.getResultModelByEObject(newAlternativeEvaluation);
							saveDirtyModel(model);
							
							// refresh the view
							alternativeEvaluationViewer.refresh();
							alternativeEvaluationViewer.expandAll();
							
							// simplifies creation of multiple items
							evalNameInputField.forceFocus();
							evalNameInputField.selectAll();
							
						} catch (RepositoryException e1) {
							logger.warn(e1);
						}
					};
				});
				
				// listener for CLOSE button
				closeButton.addSelectionListener(new SelectionAdapter() {
					@Override
					public void widgetSelected(SelectionEvent e) {
						dialog.dispose();
					}
				});
				
				// sets dialog dimensions
			    // dialog.pack();
			    dialog.setSize(300, 500);

			    // places the dialog window under the mouse cursor
				Point cursorLoc = event.display.getCursorLocation();
				cursorLoc.x -= dialog.getSize().x/2;
				cursorLoc.y -= dialog.getSize().y/2;
			    dialog.setLocation(cursorLoc);
			    
			    dialog.open();
			}
		});
	}

	/**
	 * This is used when an AlternativeEvaluation is created or deleted.
	 * 
	 * @see QImpressAlternativeEvaluationSelectionTab#createDeleteSelectedAlternativeEvaluationButton(Composite)
	 * @see QImpressAlternativeEvaluationSelectionTab#createNewAlternativeEvaluationButton(Composite)
	 * 
	 * @param alternativeEvaluation
	 */
	private void saveDirtyModel(IQModel model) {
		try {
			model.save();
		} catch (RepositoryException e) {
			logger.error("Model " + model.getName() + " cannot be saved. Exception occured.", e);
		}
	}
	
	/**
	 * A button that deletes the selected alternative evaluation.
	 * @param parent Where the button should be installed
	 */
	private void createDeleteSelectedAlternativeEvaluationButton(Composite parent)
	{
		Button deleteButton = new Button(parent, SWT.PUSH);
        deleteButton.setText("Delete Alternative Evaluation");
        
        // installs the listener
        deleteButton.addListener(SWT.Selection, new Listener() {
			
			@Override
			public void handleEvent(Event event) {
				
				// iterate over all checked elements and delete only AlternativeEvaluation objects
				for (Object element : alternativeEvaluationViewer.getCheckedElements()) {
					if(element instanceof AlternativeEvaluation) {
						AlternativeEvaluation alternativeEvaluation = (AlternativeEvaluation) element;
						IQProject qproject = QImpressCore.getQProject( alternativeEvaluation );
						
						try {
							
							IQModel model = QImpressCore.getResultModelByEObject(alternativeEvaluation);
							qproject.getRepository().deleteAlternativeEvaluation(alternativeEvaluation);
							saveDirtyModel(model);
							
						} catch (RepositoryException e) {
							logger.warn(e);
						}
					}
				}
				
				// finally refresh the tree viewer
				alternativeEvaluationViewer.refresh();
			}
		});
	}
	
	/**
	 * Update the label text showing information of the checked item from the tree.
	 */
	private void updateCheckedItemLabelText()
	{
		Object[] checked = alternativeEvaluationViewer.getCheckedElements();
		
		if(checked.length > 0 && checked[0] instanceof AlternativeEvaluation) {
			
			AlternativeEvaluation alternativeEvaluation =  (AlternativeEvaluation) checked[0];
			
			String alternativeEvaluationName = alternativeEvaluation.getName();
			String alternativeDescription = "<unknown alternative>";
			String alternativeId = alternativeEvaluation.getAlternativeId();
			
			try {
				IQAlternative alternative =
					QImpressCore
						.getQProject(alternativeEvaluation)
						.getRepository()
						.getAlternative(alternativeId);
					
				alternativeDescription =
					"'" + alternative.getInfo().getDescription() + "'";
				
			} catch (Exception e) {}
			
			// show the text in UI
			selectionText.setText("'" + alternativeEvaluationName + "' in " + alternativeDescription);
			
		} else {
			selectionText.setText("<none>");
		}
	}
	
	@Override
	public String getName() {
		return "Q-ImPrESS Alternative Evaluation";
	}

	@Override
	public void initializeFrom(ILaunchConfiguration configuration) {
	
		String cfgAlternativeEvaluationId = "";
		String cfgProjectName = "";
		
		try {
			cfgAlternativeEvaluationId = configuration.getAttribute(SELECTED_ALTERNATIVE_EVALUATION_ID, "");
			cfgProjectName = configuration.getAttribute(SELECTED_PROJECT_NAME, "");
		} catch (CoreException e) {
			logger.warn(e);
		}
		
		// nothing to do provided that nothing is configured
		if ( cfgAlternativeEvaluationId.equals("") || cfgProjectName.equals(""))
			return;

		IQRepository repository;
		
		try {
			repository = QImpressCore.getQProject(cfgProjectName).getRepository();
			
			if(repository == null)
				return;
			
		} catch (Exception e) {
			logger.warn("Could not get repository from the project name:" + cfgProjectName);
			return;
		}
		
		try {
			
			for (AlternativeEvaluation alteval : repository.getAllAlternativeEvaluations()) {
				if( cfgAlternativeEvaluationId.equals(alteval.getId()) ) {
					alternativeEvaluationViewer.setChecked(alteval, true);
					updateCheckedItemLabelText();
					return;
				}
			}
		} catch (RepositoryException e) {
			logger.warn(e);
		}
	}

	@Override
	public void performApply(ILaunchConfigurationWorkingCopy configuration) {
		Object[] elements = alternativeEvaluationViewer.getCheckedElements();
		if (elements.length != 1 || !(elements[0] instanceof AlternativeEvaluation)) {
			configuration.setAttribute(SELECTED_ALTERNATIVE_EVALUATION_ID, "");
		} else {
			AlternativeEvaluation selectedAlternativeEvaluation = (AlternativeEvaluation) elements[0];
			
			// we need this value to identify selected (checked) item
			configuration.setAttribute(SELECTED_ALTERNATIVE_EVALUATION_ID, selectedAlternativeEvaluation.getId());
			
			// we need this value to get the repository
			String projectName = QImpressCore.getQProject(selectedAlternativeEvaluation).getProject().getName();
			configuration.setAttribute(SELECTED_PROJECT_NAME, projectName);
		}
	}
	
	@Override
	public boolean isValid(ILaunchConfiguration launchConfig) {
		if (!super.isValid(launchConfig)) return false;
		
		return getErrorMessage() == null;
	}

	@Override
	public String getErrorMessage() {
		Object[] elements = alternativeEvaluationViewer.getCheckedElements();
		
		if (elements.length != 1)
			return "Select a single alternative evaluation";
		
		if (!(elements[0] instanceof AlternativeEvaluation))
			return "Selection is not an alternative evaluation";
		
		return null;
	}

	@Override
	public void setDefaults(ILaunchConfigurationWorkingCopy configuration) {
		configuration.setAttribute(SELECTED_ALTERNATIVE_EVALUATION_ID, "");
		configuration.setAttribute(SELECTED_PROJECT_NAME, "");
	}

	@Override
	public void setSelectedAlternative(IQAlternative alternative) {
		
		selectedAlternative = alternative;
		enableFiltering(selectedAlternative != null);	
		
		alternativeEvaluationViewer.expandAll();		
		setDirty(true);
	}
	
	protected void enableFiltering(boolean enable) {
		if (enable && selectedAlternative != null) {
			QAlternativeEvaluationSelectorWidgetFactory.applyFilterFor(selectedAlternative, this.alternativeEvaluationViewer);			
		} else {
			alternativeEvaluationViewer.resetFilters();
		}
	}
}
