/**
 * 
 */
package eu.qimpress.ide.backbone.core.ui.models;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EventObject;
import java.util.HashMap;
import java.util.Map;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.command.BasicCommandStack;
import org.eclipse.emf.common.command.Command;
import org.eclipse.emf.common.command.CommandStack;
import org.eclipse.emf.common.command.CommandStackListener;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.impl.ResourceFactoryImpl;
import org.eclipse.emf.ecore.resource.impl.ResourceImpl;
import org.eclipse.emf.ecore.xmi.XMIResource;
import org.eclipse.emf.ecore.xmi.impl.XMIResourceImpl;
import org.eclipse.emf.ecore.xmi.impl.XMLParserPoolImpl;
import org.eclipse.emf.edit.domain.AdapterFactoryEditingDomain;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.edit.domain.IEditingDomainProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.widgets.Display;

import eu.qimpress.ide.backbone.core.model.AbstractApplicationModelChangeListener;
import eu.qimpress.ide.backbone.core.model.IQAlternative;
import eu.qimpress.ide.backbone.core.model.IQModel;
import eu.qimpress.ide.backbone.core.model.QImpressApplicationModelManager;
import eu.qimpress.ide.backbone.core.models.resources.DefaultResourcesPlugin;
import eu.qimpress.ide.backbone.core.ui.decorators.QModelNeedsSaveDecorator;
import eu.qimpress.ide.backbone.core.ui.decorators.QModelNeedsSaveMarkerConst;
import eu.qimpress.ide.backbone.core.ui.models.adapters.QModelsComposedAdapterFactoryProvider;

/**
 * Background model editor connected to the Common Navigator view.
 * 
 * There is one-to-one relation between alternative and this editor.
 * 
 * @author Michal Malohlava
 *  
 */
public class ShadowModelEditor extends AbstractApplicationModelChangeListener implements IEditingDomainProvider {
	
	private static final Logger logger = Logger.getLogger(ShadowModelEditor.class);
	
	protected Viewer viewer;
	
	protected IQAlternative alternative;
	
	protected AdapterFactoryEditingDomain editingDomain;
	
	private Map<URI, IQModel> modelCache = new HashMap<URI, IQModel>();
	
	public ShadowModelEditor(IQAlternative alternative) {
		this.alternative = alternative;
		
		initEditingDomain();	
		
		loadDefaultModels();
		
		reloadModels(true);
	
		// register tuned resource factory for loading all resources
		editingDomain.getResourceSet().getResourceFactoryRegistry().getExtensionToFactoryMap().put(Resource.Factory.Registry.DEFAULT_EXTENSION, new PerformantXMIResourceFactoryImpl());
		
		// FIXME rewrite surrounding code to use this adapter
		editingDomain.getResourceSet().getAdapterFactories().add(new ShadowModelEditorAdapterFactory(this));
	}
	
	public void reloadModels(boolean refresh) {
		for (IQModel model : this.alternative.getModels()) {
			/* forget result */ getModelResource(model, refresh);			
		}		
	}
	
	protected void loadDefaultModels() {
		for(String modelName : DefaultResourcesPlugin.getDefault().getAccesibleModels()) {
			URI modelURI = DefaultResourcesPlugin.getDefault().getModelURI(modelName);
			
			try {
				/*Resource res =*/ editingDomain.getResourceSet().getResource(modelURI, true);
				//res.load(null);
			} catch (Exception e) {
				logger.warn("Cannot load integrated model : " + modelURI, e);			
			}		
		}
	}

	protected void initEditingDomain() {
		// Create the command stack that will notify this editor as commands are executed.
		BasicCommandStack commandStack = new BasicCommandStack();
		
		commandStack.addCommandStackListener(new CommandStackListener() {
			
			@Override
			public void commandStackChanged(final EventObject event) {
				// create marker for file which need save
				Command mostRecentCommand = ((CommandStack) event.getSource()).getMostRecentCommand();
				if (mostRecentCommand != null) {
					// received affected IQModel
					Collection affectedObjects = mostRecentCommand.getAffectedObjects();
					final Collection<EObject> affectedEObjects = new ArrayList<EObject>(3);
					
					for (Object affectedObject : affectedObjects) {					
						// get affected resource & create marker
						if (affectedObject instanceof EObject) {		
							affectedEObjects.add((EObject) affectedObject);							
						}
					}
					
					Display display = viewer != null ? viewer.getControl().getDisplay() : Display.getDefault(); 

					// create a marker asynchronously					
					display.asyncExec( new Runnable() {
						
						@Override
						public void run() {
							IQModel[] affectedModels = new IQModel[affectedEObjects.size()];
							int c = 0;
							for (final EObject eo : affectedEObjects) {
								try {									
									URI modelURI = eo.eResource().getURI();
									markModifiedModelWorkspaceResource(modelURI);
									
									if (ShadowModelEditor.this.modelCache.containsKey(modelURI)) {
										affectedModels[c++] = ShadowModelEditor.this.modelCache.get(modelURI);
									}
								} catch (CoreException e) {
									// do nothing
									logger.debug("Cannot mark resource as modified, eObject: " + eo, e);
								}
								
								fireModelsMarkerChanged(affectedModels);
							}
						}
					});					
				}
			}
		});
		
		editingDomain = new AdapterFactoryEditingDomain(QModelsComposedAdapterFactoryProvider.getInstance().getAdapterFactory(), commandStack);
	}
	
	private void markModifiedModelWorkspaceResource(URI modelURI) throws CoreException {
		
		IPath path = new Path(modelURI.toPlatformString(true));
		IFile modelFile = ResourcesPlugin.getWorkspace().getRoot().getFile(path);
		
		modelFile.createMarker(QModelNeedsSaveMarkerConst.MARKER_ID);
		
	}
	
	public Resource getModelResource(IQModel model) {
		return getModelResource(model, false);
	}
	
	public Resource getModelResource(IQModel model, boolean refresh) {
		IFile modelFile = (IFile) model.getCorrespondingResource();
		URI resourceURI = URI.createPlatformResourceURI(modelFile.getFullPath().toOSString(), false);
		Resource res = null;
		if (modelFile.exists()) {
			res = editingDomain.getResourceSet().getResource(resourceURI, true);
			
			if (res.isLoaded() && refresh) {
				res.unload();
				
				res = editingDomain.getResourceSet().getResource(resourceURI, true);							
			}
		} else {
			res = editingDomain.getResourceSet().createResource(resourceURI);
		}
		
		cacheModel(resourceURI, model);
		
		return res;		
	}
	
	public Resource createModelResource(IQModel qModel) {
		IFile modelFile = (IFile) qModel.getCorrespondingResource();
		URI resourceURI = URI.createPlatformResourceURI(modelFile.getFullPath().toOSString(), false);
		Resource resource = editingDomain.getResourceSet().createResource(resourceURI);
		
		try {
			resource.save(null);
			
		} catch (IOException e) {				
			logger.error("Cannot create file for qModel: " + qModel, e);
			resource = null;
		}
		
		cacheModel(resourceURI, qModel);
		
		return resource;	
	}
	
	@Override
	public EditingDomain getEditingDomain() {
		return editingDomain;
	}

	public void dispose() {
		// TODO call dispose from outside 
		QImpressApplicationModelManager.getManager().getQAppModel().removeChangeListener(this);
	}
	
	@Override
	public void modelCreated(IQModel[] models) {
		updateModels(models);
	}
	
	@Override
	public void modelModified(IQModel[] models) {
		updateModels(models);
	}	
	
	@Override
	public void modelDeleted(IQModel[] models) {
		for(final IQModel model : models) {
			IFile modelFile = (IFile) model.getCorrespondingResource();
			URI resourceURI = URI.createPlatformResourceURI(modelFile.getFullPath().toOSString(), false);
			
			Resource res = editingDomain.getResourceSet().getResource(resourceURI, true);
			res.unload();
		}
	}
		
	
	protected void updateModels(final IQModel[] models) {
		final String thisAlternativeId = this.alternative.getInfo().getId();		
		
		for(final IQModel model : models) {
			IQAlternative alt = (IQAlternative) model.getParent();
			if (alt == null || alt.getInfo().getId().equals(thisAlternativeId)) {
				/* reload resource */
				getModelResource(model, true);
			}			
		}
		
		Display display = viewer != null ? viewer.getControl().getDisplay() : Display.getDefault();
		display.asyncExec(new Runnable() {
			
			@Override
			public void run() {
				for (final IQModel rs : models) {
					/* clean markers */
					try {
						rs.getCorrespondingResource().deleteMarkers(QModelNeedsSaveMarkerConst.MARKER_ID, true, IResource.DEPTH_ZERO);
						
						fireModelsMarkerChanged(models);
					} catch (CoreException e) {
						logger.debug("Cannot delete markers for model resource: " + rs, e);
					}
				}
			}
		});
	}
	
	private void fireModelsMarkerChanged(final IQModel[] models) {
		QModelNeedsSaveDecorator decorator = QModelNeedsSaveDecorator.getDecorator();
		if (decorator != null) {
			decorator.refresh(models);
		}
	}

	public Viewer getViewer() {
		return viewer;
	}

	public void setViewer(Viewer viewer) {
		this.viewer = viewer;
	}
	
	private void cacheModel(URI uri, IQModel qModel) {
		modelCache.put(uri, qModel);
	}
	
	private class PerformantXMIResourceFactoryImpl extends ResourceFactoryImpl {
		
		@Override
		public Resource createResource(URI uri) {
			XMIResource resource = new XMIResourceImpl(uri);
			
			Map<Object, Object> loadOptions = resource.getDefaultLoadOptions();
			
			loadOptions.put(XMIResource.OPTION_DEFER_ATTACHMENT, Boolean.TRUE);
			loadOptions.put(XMIResource.OPTION_DEFER_IDREF_RESOLUTION, Boolean.TRUE);
			loadOptions.put(XMIResource.OPTION_USE_DEPRECATED_METHODS, Boolean.TRUE);
			loadOptions.put(XMIResource.OPTION_USE_PARSER_POOL, new XMLParserPoolImpl());
			loadOptions.put(XMIResource.OPTION_USE_XML_NAME_TO_FEATURE_MAP, new HashMap<Object, Object>());
			
			((ResourceImpl) resource).setIntrinsicIDToEObjectMap(new HashMap<String, EObject>());
			
			return resource;
		}
				
	}

	public IQModel getCachedQModel(URI modelURI) {		
		return modelCache.get(modelURI);
	}
}
