/**
 * 
 */
package eu.qimpress.ide.backbone.project.listeners;

import java.util.Collections;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.xtext.ui.core.editor.XtextEditor;

import eu.qimpress.ide.backbone.project.adapters.IConcreteSyntaxFile;

/**
 * Listen for save on textual representation of model - changes on a given Xtext resource and update EMF model if the resource is well formatted.
 * 
 * @author Michal Malohlava
 *
 */
public class XtextResourceChangeListener implements IResourceChangeListener, IPartListener {
	
	private static final Logger logger = Logger.getLogger(XtextResourceChangeListener.class);
	
	private IConcreteSyntaxFile concreteSyntaxFile;
	
	public XtextResourceChangeListener(IConcreteSyntaxFile concreteSyntaxFile) {
		this.concreteSyntaxFile = concreteSyntaxFile;		
	}

	/* (non-Javadoc)
	 * @see org.eclipse.core.resources.IResourceChangeListener#resourceChanged(org.eclipse.core.resources.IResourceChangeEvent)
	 */
	@Override
	public void resourceChanged(IResourceChangeEvent event) {
		if (event.getType() == IResourceChangeEvent.POST_CHANGE) {
			IResourceDelta resourceDelta = event.getDelta().findMember(concreteSyntaxFile.getOutputFile().getFullPath());
			if (resourceDelta != null 
					&& resourceDelta.getResource().equals(concreteSyntaxFile.getOutputFile())
					&& (resourceDelta.getFlags() & IResourceDelta.CONTENT) == IResourceDelta.CONTENT) {
				handleFileChange();
			}
		}
	}
	
	protected void handleFileChange() {
		// locate original resource
		final Resource ecoreResource = this.concreteSyntaxFile.getOriginalResource();
		final Resource xtextResource = this.concreteSyntaxFile.getConcreteResource();
		
		try {
			// reload resource			
			xtextResource.unload();
			xtextResource.load(Collections.EMPTY_MAP);

			// check concrete model for bugs
			if (!xtextResource.getErrors().isEmpty()) {
				
				Display.getDefault().asyncExec(new Runnable() {					
					@Override
					public void run() {
						MessageDialog.openWarning(Display.getCurrent().getActiveShell(), "Propagation of changes into model", "Changes in the text editor cannot be propageted into ecore model due to errors in concrete syntax");						
					}
				});
				
				return;
			} 

			// transfer contents
			Job job = new Job("Update Ecore resource") {
				
				@Override
				protected IStatus run(IProgressMonitor monitor) {
										
					try {
						ecoreResource.unload();
						ecoreResource.getContents().addAll(xtextResource.getContents());
						logger.info("Transfering content of Xtext resource back into Ecore resource");
						ecoreResource.save(null);
					} catch (Exception e) {
						logger.error("Cannot transfer content of XText model resource back into ECore resource", e);
					}
									
					// notification of ShadowEditor to reload the resource is done via ModelChange event in the backbone infrastructure 
					
					return Status.OK_STATUS;
				}
			};
			job.setPriority(Job.SHORT);
			job.schedule();
			
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	@Override
	public void partActivated(IWorkbenchPart part) {
	}

	@Override
	public void partBroughtToTop(IWorkbenchPart part) {
	}

	@Override
	public void partClosed(IWorkbenchPart part) {
		// closing editor event
		if (part instanceof XtextEditor) {
			IEditorInput editorInput = ((XtextEditor) part).getEditorInput();
			logger.trace("Try to close editor with editorinput=" + editorInput);
			if (editorInput instanceof FileEditorInput
					&&
				((FileEditorInput) editorInput).getFile().equals(this.concreteSyntaxFile.getOutputFile())) {
				
				logger.trace("Closing editor for " + editorInput);
				// disposing Xtext file - include unregistering the listener
				logger.trace("Disposing concrete file");
				this.concreteSyntaxFile.dispose();
			}
		}
	}

	@Override
	public void partDeactivated(IWorkbenchPart part) {
	}

	@Override
	public void partOpened(IWorkbenchPart part) {				
	}
}
