/**
 * 
 */
package eu.qimpress.ide.backbone.core.model;

import org.eclipse.core.resources.IFolder;

/**
 * Backbone repository.
 * 
 * @author Petr Hnetynka
 * @author Michal Malohlava
 * @author Viliam Simko
 */
public interface IQRepository extends IQElement, IParent, IQParameterProvider, IQResultRepositoryAccess, IQGlobalModelAccess, IUsageModelAccess {

	/**
	 * Alternative containing global models (shared by all alternatives)
	 */
	public static final String GLOBAL_ALTERNATIVE_NAME = "global alternative";
	
	/**
	 * ID of global alternative will be set to well known value.
	 */
	public static final String GLOBAL_ALTERNATIVE_ID = "0";	
	
	/**
	 * Alternative which is default (active) in newly created project
	 */	
	public static final String MAIN_ALTERNATIVE_NAME = "main alternative";
	
	/**
	 * Returns an array of alternatives stored in the repository.
	 * 
	 * @return an array of alternatives
	 * @throws RepositoryException 
	 */
	IQAlternative[] listAllAlternatives() throws RepositoryException;

	/**
	 * Returns an array of top level alternatives &mdash; alternatives without a
	 * parent.
	 * 
	 * @return an array of top level alternatives
	 * @throws RepositoryException
	 */
	IQAlternative[] listTopLevelAlternatives() throws RepositoryException;

	/**
	 * Returns an alternative with the given ID.
	 * 
	 * @param id ID of the required alternative
	 * @return an alternative with the given ID
	 * @throws RepositoryException
	 */
	IQAlternative getAlternative(String id) throws RepositoryException;

	/**
	 * Returns an array of child alternatives of the given alternative.
	 * 
	 * @param alt alternative
	 * @return child alternatives or empty array
	 * @throws RepositoryException
	 */
	IQAlternative[] getChildren(IQAlternative alt) throws RepositoryException;

	/**
	 * Creates new alternative without any parent.
	 * 
	 * @param description human readable description
	 * @return new alternative
	 * @throws RepositoryException alternative cannot be created
	 */
	IQAlternative createAlternative(String description)
			throws RepositoryException;

	/**
	 * Creates new alternative without any parent and with specified id.
	 * 
	 * @param description human readable description
	 * @return new alternative
	 * @throws RepositoryException alternative cannot be created
	 */
	IQAlternative createAlternative(String description, String SpecifiedId)
			throws RepositoryException;	
	
	/**
	 * Creates new alternative with the given parent and copy all content of the
	 * parent to the new alternative.
	 * 
	 * @param parent parent alternative
	 * @param description human readable description
	 * @return new alternative
	 * @throws RepositoryException alternative cannot be created
	 */
	IQAlternative createAlternative(IQAlternative parent, String description)
			throws RepositoryException;	

	/**
	 * Creates new alternative with the given parent.
	 * 
	 * @param parent parent alternative
	 * @param description human readable description
	 * @param noCopy if <code>true</code> then the parent content is not copied
	 *        &mdash; <em>use only if you are 100% sure</em>
	 * @return new alternative
	 * @throws RepositoryException alternative cannot be created
	 */
	IQAlternative createAlternative(IQAlternative parent, String description,
			boolean noCopy) throws RepositoryException;
	
	/**
	 * Deletes a given alternative. It does not delete the alternative content
	 * and refuse to delete alternatives with children.
	 * 
	 * @param alternative alternative to be deleted
	 * @throws RepositoryException alternative cannot be deleted
	 */
	void deleteAlternative(IQAlternative alternative)
			throws RepositoryException;

	/**
	 * Deletes a given alternative.
	 * 
	 * @param alternative alternative to be deleted
	 * @param deleteContent if <code>true</code> it deletes also all models in
	 *        the alternative
	 * @param force if <code>true</code> it deletes also all child alternatives
	 *        &mdash; <em>use only if you are 100% sure</em>
	 * @throws RepositoryException alternative cannot be deleted
	 */
	void deleteAlternative(IQAlternative alternative, boolean deleteContent,
			boolean force) throws RepositoryException;
	
	/**
	 * Closes the repository.
	 * 
	 * @throws RepositoryException
	 */
	void close() throws RepositoryException;
	
	/**
	 * Is the repository closed?
	 */
	boolean isClosed();
	
	/**
	 * Returns default alternative. Is there no default alternative in
	 * repository, it returns the first one and configures it as default.
	 * 
	 * @throws RepositoryException
	 */
	IQAlternative getDefaultAlternative() throws RepositoryException;

	/**
	 * Returns Q-ImPrESS project containing this repository.
	 * 
	 * @return Q-ImPrESS project
	 */
	IQProject getQProject();
	
	/* ===== */
	IFolder getRepositoryFolder();
}
