/* $Id$ */
package eu.qimpress.ide.backbone.core.internal.model;

import java.lang.reflect.InvocationTargetException;
import java.math.BigInteger;
import java.util.LinkedList;
import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.neodatis.odb.ODB;
import org.neodatis.odb.ODBFactory;
import org.neodatis.odb.Objects;
import org.neodatis.odb.core.query.IQuery;
import org.neodatis.odb.core.query.criteria.Where;
import org.neodatis.odb.impl.core.query.criteria.CriteriaQuery;

import eu.qimpress.ide.backbone.core.model.IQAlternative;
import eu.qimpress.ide.backbone.core.model.IQAlternativeInfo;
import eu.qimpress.ide.backbone.core.model.IQModel;
import eu.qimpress.ide.backbone.core.model.IQProject;
import eu.qimpress.ide.backbone.core.model.IQRepository;
import eu.qimpress.ide.backbone.core.model.ISaveable;
import eu.qimpress.ide.backbone.core.model.RepositoryException;
import eu.qimpress.ide.backbone.core.model.RepositoryModels;
import eu.qimpress.resultmodel.AlternativeEvaluation;
import eu.qimpress.resultmodel.ResultModelFactory;
import eu.qimpress.resultmodel.ResultRepository;

/**
 * Implementation of the
 * {@link eu.qimpress.ide.backbone.core.model.IQRepository} interface. Stores
 * data in a directory &mdash; each alternative in a separated subdirectory.
 * 
 * @author Petr Hnetynka
 * @author Viliam Simko
 */
public class QDirectoryRepositoryImpl extends QElement implements IQRepository,
		ISupportSaveable {

	/** Logger. */
	final static Logger logger = Logger
			.getLogger(QDirectoryRepositoryImpl.class);

	/** Default repository directory. */
	public static final String DEFAULT_REPOSITORY_LOCATION = "alternatives";

	/** Default location of the repository database. */
	static final String DEFAULT_REPOSITORY_DB_LOCATION = ".db";

	/** Directory of the repository. */
	private IFolder directory;
	
	/** Directory of the repository. */
	private IQProject qProject;	

	/** Internal database for storing alternative information. */
	private ODB odb;

	/** Whether the database is closed. */
	private boolean closed = true;
	
	/** Simple handle to a database file which needs to be locally refreshed */
	private IFile dbFile;

	/**
	 * Constructor.
	 * 
	 * @param dir
	 *            directory of the repository
	 */
	QDirectoryRepositoryImpl(IQProject project, final IFolder directory)
			throws RepositoryException {
		super(project);
		
		this.qProject = project;
		this.directory = directory;
		this.dbFile = this.directory.getFile(DEFAULT_REPOSITORY_DB_LOCATION);
		
		open();
	}
	
	protected void open() throws RepositoryException {
		WorkspaceModifyOperation initRepoOperation = new WorkspaceModifyOperation() {
			
			@Override
			protected void execute(IProgressMonitor monitor) throws CoreException,
					InvocationTargetException, InterruptedException {
				try {
					if (!directory.exists()) {					
						directory.create(true, true, null);
						logger.info("Alternatives folder created: " + directory);				
					}
					
					// open db file - this call has to be synchronized
					if (closed) {
							String filename = QDirectoryRepositoryImpl.this.dbFile.getLocation().toOSString();
							QDirectoryRepositoryImpl.this.odb = ODBFactory.open(filename);
							QDirectoryRepositoryImpl.this.closed = false;						
					}					
				} finally {
					monitor.done();
				}
			}
		};		
				

		try {
			initRepoOperation.run(null);
		} catch (Exception ex) {
			throw new RepositoryException("Database cannot be opened.", ex);
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IQAlternative[] listAllAlternatives() throws RepositoryException {
		if (closed) {
			throw new IllegalStateException("Repository is closed.");
		}

		Objects<QAlternativeInfoImpl> obs = odb
				.getObjects(QAlternativeInfoImpl.class);

		return updateRetrievedAlternatives(obs, null);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @throws RepositoryException
	 */
	@Override
	public IQAlternative[] listTopLevelAlternatives()
			throws RepositoryException {
		if (closed) {
			throw new IllegalStateException("Repository is closed.");
		}

		IQuery q = new CriteriaQuery(QAlternativeInfoImpl.class, 
				Where .and().
						add(Where.isNull("parent")).
						add(Where.not(Where.equal("id", IQRepository.GLOBAL_ALTERNATIVE_ID))
				));
		Objects<QAlternativeInfoImpl> obs = odb.getObjects(q);

		return updateRetrievedAlternatives(obs, null);
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @throws RepositoryException
	 */
	@Override
	public IQAlternative getAlternative(String id) throws RepositoryException {
		if (closed) {
			throw new IllegalStateException("Repository is closed.");
		}

		IQuery q = new CriteriaQuery(QAlternativeInfoImpl.class, Where.equal("id", id));
		Objects<QAlternativeInfoImpl> obs = odb.getObjects(q);
		
		if (obs.size() == 0) {
			return null;
		}
		
		return updateRetrievedAlternative(obs.getFirst());
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @throws RepositoryException
	 */
	@Override
	public IQAlternative[] getChildren(IQAlternative alt)
			throws RepositoryException {
		IQuery q = new CriteriaQuery(QAlternativeInfoImpl.class, Where.equal(
				"parentId", alt.getInfo().getId()));
		Objects<QAlternativeInfoImpl> obs = odb.getObjects(q);

		return updateRetrievedAlternatives(obs, alt);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IQAlternative createAlternative(String description)
			throws RepositoryException {
		return createAlternative(null, description, false, null);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IQAlternative createAlternative(IQAlternative parent,
			String description) throws RepositoryException {
		return createAlternative(parent, description, false);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IQAlternative createAlternative(IQAlternative parent,
			String description, boolean noCopy) throws RepositoryException {
		return createAlternative(parent, description, noCopy, null);
	}
	
	public IQAlternative createAlternative(String description, 
			String SpecifiedId)	throws RepositoryException{
		return createAlternative(null, description, false, IQRepository.GLOBAL_ALTERNATIVE_ID);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public IQAlternative createAlternative(IQAlternative parent,
			String description, boolean noCopy, String SpecifiedId) 
			throws RepositoryException {
		if (closed) {
			throw new IllegalStateException("Repository is closed.");
		}
		IQAlternativeInfo parentInfo = parent != null ? parent.getInfo() : null;

		IQAlternativeInfo alternativeInfo = new QAlternativeInfoImpl( parentInfo, 
				description );		
		
		if (SpecifiedId != null){
			alternativeInfo = new QAlternativeInfoImpl( parentInfo, description, 
				SpecifiedId );
		}

		IQAlternative alt = new QAlternativeImpl(this, parent, alternativeInfo,
				noCopy, true);

		odb.store(alternativeInfo);
		odb.commit();
		
		postCommit();		
		
		return alt;
	}
	

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void deleteAlternative(IQAlternative alternative)
			throws RepositoryException {
		deleteAlternative(alternative, false, false);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void deleteAlternative(IQAlternative alternative,
			boolean deleteContent, boolean force) throws RepositoryException {
		if (closed) {
			throw new IllegalStateException("Repository is closed.");
		}
		IQAlternative[] children = getChildren(alternative);
		if (force) {
			for (IQAlternative chAlt : children) {
				deleteAlternative(chAlt, deleteContent, force);
			}
		} else { // do not delete children
			if (children.length != 0) {
				throw new RepositoryException(
						"Cannot be deleted - there are child alternatives");
			}
		}
		odb.delete(alternative.getInfo());
		odb.commit();
		postCommit();
		
		if (deleteContent) {
			try {
				alternative.getAlternativeFolder().delete(true, false, null);
			} catch (CoreException ex) {
				throw new RepositoryException(
						"Cannot delete content of the alternative \""
								+ alternative.getInfo().getId() + "\"", ex);
			}
		}
	}
	
	@Override
	public void deleteAlternativeEvaluation(
			AlternativeEvaluation alternativeEvaluation)
			throws RepositoryException {

		getResultRepository().getAnalysisRuns().remove(alternativeEvaluation);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void close() throws RepositoryException {
		if (odb != null && !odb.isClosed()) {
			try {
				odb.close();
				odb = null;
			} catch (Exception ex) {
				throw new RepositoryException("Database cannot be closed.", ex);
			}
			closed = true;
		}
	}

	@Override
	public boolean isClosed() {
		return (closed || odb == null || odb.isClosed());
	}

	/**
	 * Returns a directory where the repository data are stored.
	 * 
	 * @return a directory where the repository data are stored
	 */

	void storeUpdatedAlternative(QAlternativeImpl alt) {
		odb.store(alt);
	}

	private IQAlternative[] updateRetrievedAlternatives(
			Objects<QAlternativeInfoImpl> obs, IQAlternative parentAlternative)
			throws RepositoryException {
		IQAlternative[] alternatives = new IQAlternative[obs.size()];

		int i = 0;
		for (QAlternativeInfoImpl ai : obs) {
			alternatives[i++] = new QAlternativeImpl(this, parentAlternative,
					ai, false, false);
		}

		return alternatives;
	}
	
	private IQAlternative updateRetrievedAlternative(QAlternativeInfoImpl alternativeInfo) throws RepositoryException {
		IQAlternative parentAlternative = null;
		
		String parentAlternativeId = alternativeInfo.getParentId();
		if (parentAlternativeId != null && !parentAlternativeId.equals("")) {
			parentAlternative = getAlternative(parentAlternativeId);
		}
		
		return new QAlternativeImpl(this, parentAlternative, alternativeInfo, false, false);
	}

	@Override
	public ElementType getElementType() {
		return ElementType.Q_REPOSITORY;
	}

	@Override
	public IResource getCorrespondingResource() {
		return getRepositoryFolder();
	}

	@Override
	public IFolder getRepositoryFolder() {
		return this.directory;
	}

	@Override
	public IQAlternative getDefaultAlternative() throws RepositoryException {
		IQAlternative[] alternatives = listAllAlternatives();
		IQAlternative result = null;

		for (IQAlternative alternative : alternatives) {
			if (alternative.getInfo().isDefault()) {
				result = alternative;
				break;
			}
		}

		// make the first one default
		if (result == null && alternatives.length > 0) {
			result = alternatives[0];
			result.getInfo().setDefault(true);
		}

		if (result != null) {
			// re-parent to IQProject
			((QElement) result).parent = this.parent;
		}

		return result;
	}
	
	@Override
	public IQAlternative getGlobalAlternative() throws RepositoryException {
		
		IQAlternative globalAlternative = null;
		
		try {
			globalAlternative = getAlternative(GLOBAL_ALTERNATIVE_ID);
		} catch (RepositoryException e) {
			logger.debug("Exception occured during getting global alternative", e);		
		}
		

		if (globalAlternative == null)
			throw new RepositoryException("Global alternative not found, there should always be a single global alternative in a project. To avoid this exception disable and enable Q-I nature for project: " + this.getQProject().getProject().getName());
		
		return globalAlternative;
	}

	@Override
	public void save(ISaveable o) throws RepositoryException {
		odb.store(o);
		odb.commit();
		postCommit();
	}

	@Override
	public IQProject getQProject() {
		return (IQProject) getParent();
	}

	/* ========== IQParameterProvider interface methods ======== */
	@Override
	public boolean containsKey(String key) {
		CriteriaQuery query = new CriteriaQuery(QRepositoryParam.class, Where
				.equal("key", key));
		return odb.count(query).compareTo(new BigInteger("0")) > 0;
	}

	@Override
	public Object getParameter(String key) {
		IQuery query = new CriteriaQuery(QRepositoryParam.class, Where.equal(
				"key", key));
		Objects<Object> objects = odb.getObjects(query);
		if (objects.size() == 0)
			return null;

		return ((QRepositoryParam) objects.getFirst()).value;
	}

	@Override
	public void setParameter(String key, Object value) {
		IQuery query = new CriteriaQuery(QRepositoryParam.class, Where.equal(
				"key", key));
		Objects<Object> objects = odb.getObjects(query);
		if (objects.size() == 0) {
			if (value != null)
				odb.store(new QRepositoryParam(key, value));
		} else {
			QRepositoryParam param = (QRepositoryParam) objects.getFirst();
			if (value == null)
				odb.delete(param);
			else {
				param.value = value;
				odb.store(param);
			}
		}

		odb.commit();
		
		postCommit();
	}

	/* ========== IQResultRepositoryAccess interface methods ======== */

	@Override
	public AlternativeEvaluation createAlternativeEvaluation() throws RepositoryException {		
		AlternativeEvaluation newAlternativeEvaluation = ResultModelFactory.eINSTANCE.createAlternativeEvaluation();
		// store create AlternativeEvalutation into result repository -> it needs explicit save
		getResultRepository().getAnalysisRuns().add(newAlternativeEvaluation);
		
		return newAlternativeEvaluation;
	}
	
	@Override
	public AlternativeEvaluation createAlternativeEvaluation(String alternativeId) throws RepositoryException {
		AlternativeEvaluation newAltEval = createAlternativeEvaluation(); 
		newAltEval.setAlternativeId(alternativeId);
		
		return newAltEval;
	}

	@Override
	public List<AlternativeEvaluation> getAllAlternativeEvaluations() throws RepositoryException {
		return getResultRepository().getAnalysisRuns();
	}

	@Override
	public List<AlternativeEvaluation> getAlternativeEvaluationsByAlternativeId(String id) throws RepositoryException {
		
		// aggregated list of matching items
		List<AlternativeEvaluation> listFound = new LinkedList<AlternativeEvaluation>();
		
		for (AlternativeEvaluation altEval : getAllAlternativeEvaluations()) {
			
			// filter on alternativeId
			if(id.equals( altEval.getAlternativeId() ))
				listFound.add(altEval);
		}
		
		return listFound;
	}

	@Override
	public IQModel getResultModel() throws RepositoryException {
		IFile resultModelFile = getResultRepositoryFile();
		IQModel qModel = new QModelImpl( resultModelFile, getGlobalAlternative() );

		return qModel;		
	}
	
	@Override
	public IQModel getUsageModel() throws RepositoryException {
		IFile resultModelFile = getUsageModelFile();
		IQModel qModel = new QModelImpl( resultModelFile, getGlobalAlternative() );

		return qModel;		
	}	
	
	@Override
	public ResultRepository getResultRepository() throws RepositoryException {

		IQModel resultModel = getResultModel();

		ResultRepository resultRepository = resultModel.getTopLevelEObject(
				ResultRepository.class, ResultModelFactory.eINSTANCE
						.getResultModelPackage().getResultRepository());

		return resultRepository;
	}
	
	/**
	 * Returns a URI that represents a valid default result repository XMI file.
	 * The XMI file is located in the "alternatives" directory within the
	 * project.
	 * 
	 * Note: I couldn't find any better way of converting a filename to the URI.
	 * Feel free to correct this implementation if necessary.
	 */
	protected IFile getResultRepositoryFile() {
		
		try {
			IFolder defAlternativeFolder = qProject.getRepository().getGlobalAlternative().getAlternativeFolder();
			return defAlternativeFolder.getFile(RepositoryModels.GLOBAL_REPOSITORY_MODEL_NAME + "."
					+ RepositoryModels.RESULT_MODEL_EXT);
		} catch (Exception e){
			return null;
		}
	}
	
	
	
	/**
	 * Returns a URI that represents a valid default usage model XMI file.
	 * The XMI file is located in the "alternatives" directory within the
	 * project.
	 * 
	 * Note: I couldn't find any better way of converting a filename to the URI.
	 * Feel free to correct this implementation if necessary.
	 */
	protected IFile getUsageModelFile() {
		
		try {
			IFolder defAlternativeFolder = qProject.getRepository().getGlobalAlternative().getAlternativeFolder();
			return defAlternativeFolder.getFile(RepositoryModels.USAGE_MODEL_NAME + "."
					+ RepositoryModels.USAGE_MODEL_EXT);
		} catch (Exception e){
			return null;
		}
	}	

	@Override
	public IQModel getGlobalModel(String modelType) throws RepositoryException {		
		IQAlternative alternative = getGlobalAlternative();
								
		return alternative.getModel(modelType); 
	}
	
	private void postCommit() {
		/* refresh local resource */
		try {
			dbFile.refreshLocal(IResource.DEPTH_ZERO, null);
		} catch (CoreException e) {
			logger.warn("Cannot refresh database file", e);
		}
	}
}
