package eu.qimpress.ide.backbone.core.ui.widgets;

import org.apache.log4j.Logger;
import org.eclipse.emf.edit.ui.provider.AdapterFactoryLabelProvider;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.model.WorkbenchContentProvider;

import eu.qimpress.ide.backbone.core.model.IQAlternative;
import eu.qimpress.ide.backbone.core.model.IQApplicationModel;
import eu.qimpress.ide.backbone.core.model.IQProject;
import eu.qimpress.ide.backbone.core.ui.QAppNavigatorLabelProvider;
import eu.qimpress.ide.backbone.core.ui.internal.ChainedLabelProvider;
import eu.qimpress.ide.backbone.core.utils.SAMMModelsUtils;
import eu.qimpress.ide.backbone.core.utils.SAMMModelsUtils.EnumModelLookup;
import eu.qimpress.samm.usagemodel.UsageModel;
import eu.qimpress.samm.usagemodel.provider.UsagemodelItemProviderAdapterFactory;

/**
 * Factory for creation usage models viewer widget.
 * 
 * Widget requires alternative to be selected.
 *  
 * @author Michal Malohlava
 *
 */
public class UsageModelViewerFactory extends AbstractWidgetFactory<CheckboxTreeViewer> {
	
	private static final int TREE_DEFAULT_STYLE = SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER | SWT.VIRTUAL;
	
	protected UsageModelViewerFactory(CheckboxTreeViewer widget) {
		super(widget);
	}
	
	/**
	 * Create a new widget for given parent composite widget. 
	 * 
	 * The created widget is used for selecting usage models.
	 * 
	 * @param parent
	 * @return
	 */
	public static UsageModelViewerFactory createWidgetFactory(Composite parent) {
		UsageModelViewerFactory factory = createWidgetFactory(parent, TREE_DEFAULT_STYLE, UsageModelViewerFactory.class, CheckboxTreeViewer.class);
		
		return factory;
	}
	
	/**
	 * Filter shown content to the content of the selected alternative. 
	 * 
	 * The treeViewer has to be produced by this factory, else this call has no effect.
	 * 
	 * @param alternative selected alternative
	 * @param treeViewer treeviewer to be filtered
	 * 
	 * @see EnumModelLookup
	 */
	public static void applyFilterFor(final IQAlternative alternative, CheckboxTreeViewer treeViewer) {
		treeViewer.setFilters(new ViewerFilter[] {
				new ViewerFilter() {
					
					@Override
					public boolean select(Viewer viewer, Object parentElement, Object element) {
						
						if (element instanceof IQProject ) {
							if ( ((IQProject) element).equals(alternative.getRepository().getQProject())) {
								return true;
							}
						} else if (element instanceof IQAlternative) {
							if ( ((IQAlternative) element).equals(alternative)) {
								return true;
							}							
						} else if (element instanceof UsageModel) {
							return true;
						} else if (element instanceof String) {
							return true;
						}
							
						return false;
					}
				}				
		});
	}
	
	/**
	 * Setup model lookup mode for a given tree viewer. 
	 * 
	 * The treeViewer has to be produced by this factory, else this call has no effect.
	 * 
	 * @param lookup model lookup mode
	 * @param treeViewer
	 * 
	 * @see EnumModelLookup
	 */
	public static void setModelLookup(EnumModelLookup lookup, CheckboxTreeViewer treeViewer) {
		IContentProvider provider = treeViewer.getContentProvider();
		
		if (provider instanceof UsageModelNavigatorContentProvider) {
			((UsageModelNavigatorContentProvider) provider).setModelLookup(lookup);
		}
	}
	
	public static EnumModelLookup getModelLookup(CheckboxTreeViewer treeViewer) {
		IContentProvider provider = treeViewer.getContentProvider();
		
		if (provider instanceof UsageModelNavigatorContentProvider) {
			return ((UsageModelNavigatorContentProvider) provider).getModelLookup();
		}
		
		return EnumModelLookup.ONLY_COMMON;
	}
	
	/**
	 * Setup flat view mode for given treeViewer.
	 * 
	 * The treeViewer has to be produced by this factory, else this call has no effect.
	 * 
	 * @param flatMode if true only models of selected alternative are shown else alternative's project is shown.
	 * 
	 * @param treeViewer
	 */
	public static void setFlatViewMode(boolean flatMode, CheckboxTreeViewer treeViewer) {
		IContentProvider provider = treeViewer.getContentProvider();
		
		if (provider instanceof UsageModelNavigatorContentProvider) {
			((UsageModelNavigatorContentProvider) provider).setFlatViewMode(flatMode);
		}
		
	}

	@Override
	protected void configureWidget() {
		// setup content providers
		this.widget.setContentProvider(new UsageModelNavigatorContentProvider());		
		this.widget.setLabelProvider(new ChainedLabelProvider( 
											new QAppNavigatorLabelProvider(),
											new ChainedLabelProvider(
													new AdapterFactoryLabelProvider(new UsagemodelItemProviderAdapterFactory())
											)
									));
		
		this.widget.setInput(null);
	}

	@Override
	protected void finalizeWidget() {
		// add selection listener
		this.widget.addCheckStateListener(new ICheckStateListener() {
			public void checkStateChanged(CheckStateChangedEvent event) {
				Object element = event.getElement();			
														
				if (element instanceof UsageModel) {						
					if (event.getChecked() == false) {
						widget.setCheckedElements(new Object[] {});
						
					} else {
						widget.setCheckedElements(new Object[] { event.getElement() });
					}
				} else {
					widget.setChecked(element, false);
				}				
			}
		});		
	}
	
	public UsageModelViewerFactory setupModelLookupMode(EnumModelLookup lookup) {
		( (UsageModelNavigatorContentProvider) this.widget.getContentProvider()).setModelLookup(lookup);
		
		return this;
	}
	
	public static class UsageModelNavigatorContentProvider extends WorkbenchContentProvider {
		
		private static final Object[] NO_CHILDREN = new Object[0];
		
		private static final Logger logger = Logger.getLogger(UsageModelNavigatorContentProvider.class);
		
		private EnumModelLookup modelLookup = EnumModelLookup.ONLY_COMMON;
		private boolean flatViewMode = true;
		
		@Override
		public Object[] getChildren(Object element) {
			try {
				if (element instanceof IQProject) {	
					return ((IQProject) element).getRepository().listAllAlternatives();
				} else if (element instanceof IQAlternative) {
					return SAMMModelsUtils.getUsageModels((IQAlternative) element, modelLookup);
				}
			} catch (Exception e) {
				logger.error("Cannot get children for " + element, e);
			}
			
			return super.getChildren(element);
		}
		
		@Override
		public Object[] getElements(Object element) {
			if (element instanceof IQApplicationModel) {
				IQApplicationModel qAppModel = (IQApplicationModel) element;
				
				return qAppModel.getQProjects();
			} else if (element instanceof IQAlternative) {
				if (flatViewMode) {
					return this.getChildren(element);
				} else {
					return new Object[] { ((IQAlternative) element).getRepository().getQProject() };					
				}
			} else if (element == null) {
				return NO_CHILDREN;
			}
			
			return super.getElements(element);
		}

		public void setModelLookup(EnumModelLookup modelLookup) {
			this.modelLookup = modelLookup;
		}

		public EnumModelLookup getModelLookup() {
			return modelLookup;
		}
		
		public void setFlatViewMode(boolean flatMode) {
			this.flatViewMode = flatMode;			
		}
	}
}
