/**
 * 
 */
package eu.qimpress.ide.backbone.core.ui.widgets;

import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.IElementComparer;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;

import eu.qimpress.ide.backbone.core.QImpressCore;
import eu.qimpress.ide.backbone.core.model.IQAlternative;
import eu.qimpress.ide.backbone.core.model.IQElement;
import eu.qimpress.ide.backbone.core.model.IQModel;
import eu.qimpress.ide.backbone.core.model.IQProject;
import eu.qimpress.ide.backbone.core.model.IQRepository;
import eu.qimpress.ide.backbone.core.model.QImpressApplicationModelManager;
import eu.qimpress.ide.backbone.core.ui.QAppNavigatorContentProvider;
import eu.qimpress.ide.backbone.core.ui.QAppNavigatorLabelProvider;
import eu.qimpress.ide.backbone.core.ui.QAppNavigatorContentProvider.ContentFilterEnum;

/**
 * Widget factory producing different types of tree viewers according to a specified parameters.
 * 
 * @author Michal Malohlava
 *
 */
public class QAlternativesTreeViewerFactory {
	
	/**
	 * Enum specifying the behavior of the alternatives viewer.  
	 *  
	 * @author Michal Malohlava
	 *
	 */
	public enum SelectAlternativesEnum {
		NONE, 
		SELECT_ONE, 
		SELECT_MULTIPLE
	}
	
	private TreeViewer viewer;
	private SelectAlternativesEnum selectAlternatives;
	
	private static final IElementComparer ELEMENT_COMPARER = new QElementComparer();
	
	private QAlternativesTreeViewerFactory(TreeViewer viewer, SelectAlternativesEnum selectAlternatives) {
		this.viewer = viewer;
		this.selectAlternatives = selectAlternatives;
	}
	
	public static QAlternativesTreeViewerFactory createTreeViewer(Composite parent, SelectAlternativesEnum selectAlternatives) {
		TreeViewer alternativeViewer = null;
		
		if (selectAlternatives == SelectAlternativesEnum.NONE) {
			alternativeViewer = new QAlternativesTreeViewer(
					parent,
					SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER | SWT.VIRTUAL);
		} else {
			
			alternativeViewer = new QAlternativesCheckBoxTreeViewer(
					parent,
					SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER | SWT.VIRTUAL, 
					selectAlternatives == SelectAlternativesEnum.SELECT_ONE ? true : false);
		}
		
		// setup content and label providers
		alternativeViewer.setContentProvider(new QAppNavigatorContentProvider(
				ContentFilterEnum.FILTER_DEFAULT_ALTERNATIVE.val()));
		alternativeViewer.setLabelProvider(new QAppNavigatorLabelProvider());
		
		// set input
		alternativeViewer.setInput(QImpressApplicationModelManager.getManager().getQAppModel());
		
		// modify element comparison
		alternativeViewer.setComparer(ELEMENT_COMPARER);
		
		// filter-out some elements from the tree viewer
		alternativeViewer.addFilter( new ViewerFilter() {
			
			@Override
			public boolean select(Viewer viewer, Object parentElement, Object element) {
				
				// remove IQmodel
				if(element instanceof IQModel)
					return false;
						
				// remove "global alternative" from view
				if(element instanceof IQAlternative && QImpressCore.isGlobalAlternative( ((IQAlternative) element) ))
					return false;

				return true;
			}
		});
		
		return new QAlternativesTreeViewerFactory(alternativeViewer, selectAlternatives);
	}
	
	/**
	 * 
	 * @param parentAlternative alternative which should be selected in the tree, can be null.
	 */
	public QAlternativesTreeViewerFactory setParentAlternative(IQAlternative parentAlternative) {
		setParentAlternativeInternal(parentAlternative);
		
		return this;
	}
	
	public QAlternativesTreeViewerFactory setParentAlternative(IQRepository repository) {
		setParentAlternativeInternal(repository);
		
		return this;
	}
	
	public QAlternativesTreeViewerFactory setCanSelectQRepository(boolean canSelect) {
		switch (selectAlternatives) {
		case NONE:			
			break;
		case SELECT_ONE:
		case SELECT_MULTIPLE:
			((QAlternativesCheckBoxTreeViewer) viewer).setCanSelectRepository(canSelect);
			break;
		default:
			break;
		}		
		return this;
	}
	
	private void setParentAlternativeInternal(IQElement element) {
		if (element != null) {
			viewer.expandToLevel(element, TreeViewer.ALL_LEVELS);
						
			viewer.setSelection(new StructuredSelection(element), true);			
			
			if (this.selectAlternatives != SelectAlternativesEnum.NONE) {
				((CheckboxTreeViewer) this.viewer).setChecked(element, true);
			}
		}
	}
	
	private void  setParentElementInternal(IQElement element) {
		if (element != null) {
			
			//Alternatives are on level 1..
			viewer.expandToLevel(element, 1);
						
			viewer.setSelection(new StructuredSelection(element), true);			
			
			if (this.selectAlternatives != SelectAlternativesEnum.NONE) {
				((CheckboxTreeViewer) this.viewer).setChecked(element, true);
			}
		}else {
		}
	}	
	
	public QAlternativesTreeViewerFactory   setParentElement(IQElement element) {
		if (element instanceof IQAlternative){
			setParentAlternative((IQAlternative)element);
		}else {
			setParentElementInternal(element);
		}
		return this;
	}	
	
	public TreeViewer getTreeViewer() {
		return this.viewer;
	}
	
	private static class QElementComparer implements IElementComparer {

		@Override
		public boolean equals(Object a, Object b) {			
			if (a == b)
				return true;
			if (a == null || b == null)
				return false;
			
			if (a instanceof IQProject) {
				a = ((IQProject) a).getProject();
			}
			
			if (b instanceof IQProject) {
				b = ((IQProject) b).getProject();
			}
			
			return a.equals(b);
		}

		@Override
		public int hashCode(Object element) {			
			if (element instanceof IQProject) {
				return ((IQProject) element).getProject().hashCode();
			}
			
			return element.hashCode();
		}
		
	}

}
