/**
 * 
 */
package eu.qimpress.ide.backbone.core.ui.tabs;

import org.apache.log4j.Logger;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTab;
import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;

import eu.qimpress.ide.backbone.core.QImpressCore;
import eu.qimpress.ide.backbone.core.model.IQAlternative;
import eu.qimpress.ide.backbone.core.model.IQProject;
import eu.qimpress.ide.backbone.core.model.QImpressApplicationModelManager;
import eu.qimpress.ide.backbone.core.model.RepositoryException;
import eu.qimpress.ide.backbone.core.ui.widgets.UsageModelViewerFactory;
import eu.qimpress.ide.backbone.core.utils.SAMMModelsUtils;
import eu.qimpress.ide.backbone.core.utils.SAMMModelsUtils.EnumModelLookup;
import eu.qimpress.samm.usagemodel.UsageModel;

/**
 * Launch configuration tab for selecting usage models.
 * 
 * It allows enforcing the alternative, then it show a usage models in the selected alternative.
 * 
 * @author Michal Malohlava
 *
 */
public class QImpressUsageModelSelectionTab extends AbstractLaunchConfigurationTab implements IAlternativeSelection {
	public static final String SELECTED_USAGE_MODEL_ID = "eu.qimpress.ide.backbone.core.ui.UsageModelSelection.id";
	public static final String SELECTED_USAGE_MODEL_PROJECT_NAME = "eu.qimpress.ide.backbone.core.ui.UsageModelSelection.project";
	public static final String SELECTED_USAGE_MODEL_ALTERNATIVE_ID = "eu.qimpress.ide.backbone.core.ui.UsageModelSelection.alternative.id";
	
	private static final Logger logger = Logger.getLogger(QImpressUsageModelSelectionTab.class);
	
	private static final boolean DEFAULT_VIEW_MODE_FLAT = true; 
	
	private CheckboxTreeViewer usageModelViewer;
	private Label selectionText;
	private boolean usageModelsFlatMode = DEFAULT_VIEW_MODE_FLAT;
	
	private IQAlternative selectedAlternative;

	@Override
	public void createControl(Composite parent) {
		Composite myTabControl = new Composite(parent,SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		gridLayout.marginWidth = 5;
		gridLayout.marginHeight = 5;
		gridLayout.numColumns = 2;
		myTabControl.setLayout(gridLayout);
		setControl(myTabControl);
		
		Group grpSelectUsageModel = new Group(myTabControl, SWT.FILL);
		grpSelectUsageModel.setText("Select usage model");
		FillLayout fillLayout = new FillLayout(SWT.HORIZONTAL);
		fillLayout.marginWidth = 5;
		fillLayout.marginHeight = 5;
		grpSelectUsageModel.setLayout(fillLayout);
		grpSelectUsageModel.setLayoutData(new GridData(SWT.FILL,SWT.FILL,true,true,2,1));
			
		usageModelViewer = (CheckboxTreeViewer) UsageModelViewerFactory.createWidgetFactory(grpSelectUsageModel).getWidget();
		usageModelViewer.expandAll();
		usageModelViewer.addCheckStateListener(new ICheckStateListener() {
			
			/**
			 * Called when the user manually click the checkbox.
			 */
			public void checkStateChanged(CheckStateChangedEvent event) {
				setDirty(true);
				updateLaunchConfigurationDialog();
				updateCheckedItemLabelText();
			}
		});
		
		/* debug code */
		usageModelViewer.addDoubleClickListener(new IDoubleClickListener() {
			
			@Override
			public void doubleClick(DoubleClickEvent event) {
				ISelection selection = event.getSelection();
				if (selection instanceof IStructuredSelection
						&&
					!selection.isEmpty()) {
					Object selectedObject = ((IStructuredSelection) selection).getFirstElement();
					
					if (selectedObject instanceof IQAlternative) {
						if (QImpressUsageModelSelectionTab.this.usageModelViewer.getFilters().length == 0) {
							QImpressUsageModelSelectionTab.this.setSelectedAlternative((IQAlternative) selectedObject);							
						} else {
							QImpressUsageModelSelectionTab.this.setSelectedAlternative(null);							
						}
					} else /* if (selectedObject instanceof IQProject)*/ {
						QImpressUsageModelSelectionTab.this.setSelectedAlternative(null);						
					}
				}
				
			}
		});
		/* end of debug code */
		
		Label selection = new Label(myTabControl,SWT.NONE);
		selection.setText("Selected usage model: ");
		selection.setLayoutData(new GridData(SWT.FILL,SWT.FILL,false,false,1,1));
		
		selectionText = new Label(myTabControl,SWT.NONE);
		selectionText.setText("<none>");
		selectionText.setLayoutData(new GridData(SWT.FILL,SWT.FILL,false,false,1,1));		
	}

	protected void updateCheckedItemLabelText() {
		Object[] checkedUsageModel = usageModelViewer.getCheckedElements();
		
		if (checkedUsageModel.length > 0 && checkedUsageModel[0] instanceof UsageModel) {
			selectionText.setText( ((UsageModel) checkedUsageModel[0]).getName());			
		} else {
			selectionText.setText("<none>");
		}
	}

	@Override
	public String getName() {		
		return "Q-ImPrESS Usage Model";
	}

	@Override
	public void initializeFrom(ILaunchConfiguration configuration) {	
		
		String cfgUsageModelId = null;
		String cfgAlternativeId = null;
		String cfgProjectName = null;		
		
		try {
			cfgUsageModelId = configuration.getAttribute(SELECTED_USAGE_MODEL_ID, (String) null);
			cfgAlternativeId = configuration.getAttribute(SELECTED_USAGE_MODEL_ALTERNATIVE_ID, (String) null);
			cfgProjectName = configuration.getAttribute(SELECTED_USAGE_MODEL_PROJECT_NAME, (String) null);
			
			if (cfgUsageModelId != null && cfgAlternativeId != null && cfgProjectName != null) {
				if (selectedAlternative != null && !selectedAlternative.getInfo().getId().equals(cfgAlternativeId)) {
					setUsageModelInView(this.selectedAlternative, cfgUsageModelId);					
				} else if (selectedAlternative == null && !this.usageModelsFlatMode) {
					initializeFormInternal(cfgUsageModelId, cfgAlternativeId, cfgProjectName);
				}
			}
		} catch (Exception e) {
			logger.warn("Cannot initialize usage model selection form", e);
		}		
	}
	
	protected void initializeFormInternal(String usageModelId, String alternativeId, String projectName) throws RepositoryException {
		IQProject project = QImpressCore.getQProject(projectName);		
		if (project == null) return;
		IQAlternative alternative = project.getRepository().getAlternative(alternativeId);
		if (alternative == null) return;
		
		setUsageModelInView(alternative, usageModelId);
	}
	
	protected void setUsageModelInView(IQAlternative alternative, String usageModelId) throws RepositoryException {
		EnumModelLookup lookup = UsageModelViewerFactory.getModelLookup(this.usageModelViewer);
		UsageModel[] usageModels = SAMMModelsUtils.getUsageModels(alternative, lookup);
				
		for (UsageModel usageModel : usageModels) {
			if (usageModel.getId().equals(usageModelId)) {
				this.setSelectedAlternative(alternative);
				this.usageModelViewer.setChecked(usageModel, true);					
				break;
			}								
		}
	}

	@Override
	public void performApply(ILaunchConfigurationWorkingCopy configuration) {
		Object[] elements = usageModelViewer.getCheckedElements();
		if (elements.length != 1 || !(elements[0] instanceof UsageModel)) {
			configuration.setAttribute(SELECTED_USAGE_MODEL_ID, "");
		} else {
			UsageModel selectedUsageModel = (UsageModel) elements[0];
			
			// we need this value to identify selected (checked) item
			configuration.setAttribute(SELECTED_USAGE_MODEL_ID, selectedUsageModel.getId());
			configuration.setAttribute(SELECTED_USAGE_MODEL_ALTERNATIVE_ID, QImpressCore.getAlternativeId(selectedUsageModel.eResource().getURI()));									
			// we need this value to get the repository
			String projectName = QImpressCore.getQProject(selectedUsageModel).getProject().getName();
			configuration.setAttribute(SELECTED_USAGE_MODEL_PROJECT_NAME, projectName);
		}		
	}

	@Override
	public void setDefaults(ILaunchConfigurationWorkingCopy configuration) {
		configuration.setAttribute(SELECTED_USAGE_MODEL_ID, (String) null);
		configuration.setAttribute(SELECTED_USAGE_MODEL_ALTERNATIVE_ID, (String) null);
		configuration.setAttribute(SELECTED_USAGE_MODEL_PROJECT_NAME, (String) null);		
	}
	
	@Override
	public boolean isValid(ILaunchConfiguration launchConfig) {
		setErrorMessage(null);
		
		return validatePage();		
	}

	protected boolean validatePage() {
		Object[] selectedObject = this.usageModelViewer.getCheckedElements();
		if (selectedObject.length != 1) {
			setErrorMessage("Usage model has to be selected");
			
			return false;
		}
		
		return true;
	}

	/**
	 * Enforce selected alternative.
	 * 
	 * @param alternative 
	 */
	public void setSelectedAlternative(IQAlternative alternative) {
		// first we need to setup input
		if (usageModelsFlatMode) {
			this.usageModelViewer.setInput(alternative);
		} else {
			this.usageModelViewer.setInput(QImpressApplicationModelManager.getManager().getQAppModel());
		}
		
		// and then we can filter		
		if (alternative == null) {
			this.usageModelViewer.resetFilters();
		} else {
			UsageModelViewerFactory.applyFilterFor(alternative, this.usageModelViewer);			
		}
		
		usageModelViewer.expandAll();
		
		setDirty(true);
		
		this.selectedAlternative = alternative;
	}
	
	/**
	 * Set flat mode to view a list of usage models. If false is specified then also the parent alternative
	 * and project are shown.
	 * 
	 * @param mode true for flat mode, false for hierarchical model, flat is default
	 */
	public void setFlatViewMode(boolean mode) {
		this.usageModelsFlatMode = mode;
		
		UsageModelViewerFactory.setFlatViewMode(mode, this.usageModelViewer);		
	}
	
	/**
	 * Return selected UsageModel or null
	 * 
	 * @return selected instance of UsageModel or null if no model is selected.
	 */
	public UsageModel getSelectedUsageModel() {
		Object[] selectedObject = this.usageModelViewer.getCheckedElements();
		if (selectedObject.length == 1) {
			return (UsageModel) selectedObject[0];
		} else {
			return null;
		}		
	}
}
