/**
 * 
 */
package eu.qimpress.ide.backbone.core.ui;


import java.util.Collection;
import java.util.LinkedList;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.model.WorkbenchContentProvider;

import eu.qimpress.ide.backbone.core.QImpressCore;
import eu.qimpress.ide.backbone.core.QImpressNature;
import eu.qimpress.ide.backbone.core.model.IQAlternative;
import eu.qimpress.ide.backbone.core.model.IQApplicationModel;
import eu.qimpress.ide.backbone.core.model.IQApplicationModelChangeListener;
import eu.qimpress.ide.backbone.core.model.IQModel;
import eu.qimpress.ide.backbone.core.model.IQProject;
import eu.qimpress.ide.backbone.core.model.IQRepository;
import eu.qimpress.ide.backbone.core.model.QImpressApplicationModelManager;
import eu.qimpress.ide.backbone.core.model.RepositoryException;

/**
 * Extension for Project Explorer showing Q-ImPrESS project related nodes.
 * 
 * @author Michal Malohlava
 *
 */
public class QAppNavigatorContentProvider extends WorkbenchContentProvider
		implements IQApplicationModelChangeListener {
	
	private static final Logger logger =  Logger.getLogger(QAppNavigatorContentProvider.class); 
	
	private Viewer viewer;

	public static enum ContentFilterEnum {
		
		NO_FILTER (0),
		FILTER_REPOSITORY (1),
		FILTER_ALTERNATIVES (2),
		FILTER_DEFAULT_ALTERNATIVE (4),
		FILTER_PROJECTS_ONLY (8);
		
		private int val;
		
		private ContentFilterEnum(int val) {
			this.val = val;			
		}
		
		public boolean isSet(final int filter) {
			return ((this.val & filter) == this.val);			
		}
		
		public int val() { return this.val; }
	};
	
	private int contentFilter;
	
	private static final Object[] NO_CHILDREN = new Object[0];
	
	public QAppNavigatorContentProvider() {
		this(ContentFilterEnum.NO_FILTER.val);		
	}
	
	public QAppNavigatorContentProvider(int staticContentFilter) {
		this.contentFilter = staticContentFilter;
	}

	private void runViewerUpdate(final Runnable run) {
		Control ctrl = viewer != null ? viewer.getControl() : null;
		if (ctrl == null || ctrl.isDisposed()) {
			return;
		}
		
		if (ctrl.getDisplay().getThread() == Thread.currentThread()) {
			run.run();
		} else {
			ctrl.getDisplay().asyncExec(new Runnable(){
				/* (non-Javadoc)
				 * @see java.lang.Runnable#run()
				 */
				public void run() {
					//Abort if this happens after disposes
					Control ctrl = viewer.getControl();
					if (ctrl == null || ctrl.isDisposed()) {
						return;
					}
					
					run.run();
				}
			});
		}				
	}
	
	@Override
	public void refresh() {
		runViewerUpdate(new Runnable() {		
			@Override
			public void run() {
				if (viewer != null) viewer.refresh();
			}
		});
	}

	@Override
	public void alternativeCreated(final IQAlternative alternative) {
		runViewerUpdate(new Runnable() {		
			@Override
			public void run() {
				if (viewer != null) {
					viewer.refresh();				
					viewer.setSelection(new StructuredSelection(alternative), true);
				}
			}
		});
	}
	
	@Override
	public void modelCreated(final IQModel[] model) {
		runViewerUpdate(new Runnable() {		
			@Override
			public void run() {
				if (viewer != null) viewer.refresh();				
			}
		});		
	}
	
	@Override
	public void modelModified(IQModel[] models) {
		runViewerUpdate(new Runnable() {		
			@Override
			public void run() {
				if (viewer != null) viewer.refresh();				
			}
		});		
	}
	
	@Override
	public void modelDeleted(IQModel[] models) {
	}

	@Override
	public void inputChanged(Viewer vwr, Object oldInput, Object newInput) {
		super.inputChanged(vwr, oldInput, newInput);

		viewer = vwr;

		IQApplicationModel appModel = QImpressApplicationModelManager.getManager().getQAppModel();
		
		if (oldInput != null) {
			appModel.removeChangeListener(this);
		}
		
		if (newInput != null) {
			appModel.addChangeListener(this);
		}
	}
		
	@Override
	public Object[] getChildren(Object element) {
		if (element instanceof IProject
				&& QImpressNature.hasThisNature((IProject) element)) {
			return this.getChildren(QImpressCore.create((IProject) element));
		} else if (element instanceof IQApplicationModel) {
			return getQAppProjects((IQApplicationModel) element);
		} else if (element instanceof IQProject) {
			if (!ContentFilterEnum.FILTER_PROJECTS_ONLY.isSet(this.contentFilter)){
				return getDefAltAndRepository((IQProject) element) ; 
			}else {
				return NO_CHILDREN;
			}
		} else if (element instanceof IQRepository) {
			return getAlternatives((IQRepository) element);			
		} else if (element instanceof IQAlternative) {
			
			IQAlternative alternative = (IQAlternative) element;
			if (alternative.getInfo().isDefault() && alternative.getParent() instanceof IQProject) {
				return getModels(alternative);
			} else if (QImpressCore.isGlobalAlternative(alternative)){
				return getModels(alternative);
			} else {
				return getChildrenAlternatives(alternative);
			}
		}
		
		return super.getChildren(element);
	}
	
	@Override
	public Object[] getElements(Object element) {
		if (element instanceof IQApplicationModel) {
			IProject[] projects = ((IQApplicationModel) element).getWorkspace().getRoot().getProjects();
			Collection<IProject> result = new LinkedList<IProject>();
			for(IProject project : projects) {				
				try {
					if (project.isAccessible()){
						if (project.hasNature(QImpressNature.NATURE_ID)) {
							result.add(project);
						}
					}
				} catch (CoreException e) {
					logger.error("Cannot get elements from IQ app model.", e);
				}
			}
			
			return result.toArray(); 
		}
		
		if (element instanceof IProject) {
			IProject project = (IProject) element;
			
			try {
				if (project.hasNature(QImpressNature.NATURE_ID)) {
					return super.getElements(QImpressCore.create((IProject) element));
				}
			} catch (CoreException e) {
				logger.error("Cannot get element from project", e);
			}
		}
		
		return super.getElements(element);
	}
	
	private Object[] getAlternatives(IQRepository repository) {		
		try {	
			return repository.listTopLevelAlternatives();
		} catch (RepositoryException e) {
			logger.error("Cannot get alternatives from IQRepository", e);
			return NO_CHILDREN;
		}
	}


	private Object[] getDefAltAndRepository(IQProject element) {
		//put into view node for default Alternative and a node with hierarchy view of alternatives
		Object[] defaultAlternative = getDefaultAlternative(element);
		Object[] alternativeRepository = getRepository(element);
		Object[] resultModel = getResultModel(element);
		Object[] globalAlternative = getGlobalAlternative(element);
		Object[] result = new Object[defaultAlternative.length + alternativeRepository.length + resultModel.length + globalAlternative.length];
		
		int i = 0;
		
		for (Object o : defaultAlternative) {
			result[i++] = o;			
		}
		
		for (Object o : alternativeRepository) {
			result[i++] = o;			
		}

		for (Object o : resultModel) {
			result[i++] = o;	
		}
		
		for (Object o : globalAlternative) {
			result[i++] = o;	
		}
		
		return result;
	}

	
	private Object[] getGlobalAlternative(IQProject element) {
		IQAlternative globalAlternative = null;
		try {
			globalAlternative = element.getRepository().getGlobalAlternative();
		} catch (RepositoryException e) {			
			logger.warn("Cannot get global alternative. Ignoring. Project: " + element.getProject().getName());
		}
		
		return globalAlternative != null ? new Object[] {globalAlternative} : NO_CHILDREN;
	}

	private Object[] getQAppProjects(IQApplicationModel model) {		
		return model.getQProjects();
	}
	
	private Object[] getDefaultAlternative(IQProject project) {
		Object[] result = null;
		
		if (ContentFilterEnum.FILTER_DEFAULT_ALTERNATIVE.isSet(this.contentFilter)) {
			return NO_CHILDREN;
		}
		
		try {
			IQRepository repo = project.getRepository();
			IQAlternative defaultAlternative = repo.getDefaultAlternative();
			if (defaultAlternative != null) {
				result = new Object[] { defaultAlternative };
			} else {
				result = NO_CHILDREN;	
			}
		} catch (RepositoryException e) {
			logger.error("Cannot get default alternative for project!", e );
			result = NO_CHILDREN;
		}		
		
		return result;		
	}
	
	private Object[] getRepository(IQProject project) {
		Object[] result = NO_CHILDREN;
		
		if (ContentFilterEnum.FILTER_REPOSITORY.isSet(contentFilter)) {
			return NO_CHILDREN;
		}
		
		try {
			IQRepository repo = project.getRepository();

			if (repo != null) {
				result = new Object[] { repo };
			} else {
				result = NO_CHILDREN;
			}
		} catch (RepositoryException e) {
			logger.error("Cannot get repository for project", e);
			result = NO_CHILDREN;
		}		
		
		return result;		
	}
	
	private Object[] getModels(IQAlternative modelAlternative) {
		return modelAlternative.getModels();
	}
	
	private Object[] getResultModel(IQProject project) {
		Object[] result = null;
		
		try {
			IQModel resultModel = project.getRepository().getResultModel();
		
			if (resultModel != null) {
				result = new Object[] { resultModel };
			} else {
				result = NO_CHILDREN;	
			}
		} catch (RepositoryException e) {
			logger.error("Cannot get result model for this project project!", e );
			result = NO_CHILDREN;
		}	
		
		return result;		
	}
	
	private Object[] getChildrenAlternatives(IQAlternative alternativa) {		
		try {
			return alternativa.getRepository().getChildren(alternativa);
		} catch (RepositoryException e) {
			logger.error("Cannot get children alternative for alternative", e);
			return NO_CHILDREN;			
		}
	}	
}
