/**
 * 
 */
package eu.qimpress.ide.backbone.core.ui.models.adapters;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.EMFPlugin;
import org.eclipse.emf.common.notify.AdapterFactory;
import org.eclipse.emf.edit.provider.ComposedAdapterFactory;

/**
 * Simple factory providing access to generated EMF item providers adapter factories.
 * 
 * It look into the {@link EMFPlugin} register of item providers (registered by <code>org.eclipse.emf.edit.itemProviderAdapterFactories</code> extension point and
 * according to value of<code>onlyQImpressFactories</code> it prepares item providers adapter factories. 
 * 
 * @author Michal Malohlava
 *
 */
public class QModelsComposedAdapterFactoryProvider {

	private static final Logger logger = Logger.getLogger(QModelsComposedAdapterFactoryProvider.class);
	
	/** Default value for packages filtering */
	private static final boolean ONLY_Q_IMPRESS_FACTORIES = false; 

	/** Singleton */
	private static final QModelsComposedAdapterFactoryProvider INSTANCE = new QModelsComposedAdapterFactoryProvider(ONLY_Q_IMPRESS_FACTORIES);
	
	/** Provided adapter factory */
	private ComposedAdapterFactory adapterFactory;
	
	/** Filter Q-ImPrESS packages or not */
	private boolean onlyQImpressFactories;

	private QModelsComposedAdapterFactoryProvider(boolean filterQImpressPackages) {
		this.setOnlyQImpressFactories(filterQImpressPackages);
	}

	/**
	 * Returns an instance of factory provider.
	 * 
	 * @return instance of factory provider
	 */
	public final static QModelsComposedAdapterFactoryProvider getInstance() {
		return INSTANCE;
	}
	
	@SuppressWarnings("unchecked")
	protected List<AdapterFactory> createFactoryList() {
		Map<String, AdapterFactory> list = new HashMap<String, AdapterFactory>();
		
		logger.trace("Initialization of adapter factories");
		
		// FIXME this type-case depends on the implementation of ComposedAdapterFactory.Descriptor.Registry.Impl
		if (ComposedAdapterFactory.Descriptor.Registry.INSTANCE instanceof HashMap) {
			Set<Entry<Collection<?>, Object>> descriptors = ((HashMap<Collection<?>, Object>) ComposedAdapterFactory.Descriptor.Registry.INSTANCE).entrySet();
			
			for (Entry<Collection<?>, Object> entry : descriptors) {
				if (!isOnlyQImpressFactories() 
						||
					(isOnlyQImpressFactories() && isQImpressAdapter(entry.getKey()))
					) {
						String modelURL = getModelURL(entry.getKey());
						if (!list.containsKey(modelURL)) {
							if (entry.getValue() instanceof ComposedAdapterFactory.Descriptor) {
								try {
								
									ComposedAdapterFactory.Descriptor desc = (ComposedAdapterFactory.Descriptor) entry.getValue();
									list.put(modelURL, desc.createAdapterFactory());
								} catch (Exception e) {									
									logger.trace("Cannot create adapter factory. Skipped." , e);
								}
							}
						}
				}
			}
		}
				
		return new ArrayList<AdapterFactory>(list.values());
	}
	
	protected String getModelURL(Collection<?> key) {
		for (Object o : key) {
			if (o instanceof String 
					&& ((String) o).startsWith("http")
				) {
				return (String) o;								
			}
		}
		
		return null;
	}
	
	/**
	 * Identifies only q-impress related adapters.
	 * FIXME provide way how to declaratively (via ext. point) select q-impress related adapters
	 * and how to detect their dependencies (e.g., ecore meta-model). 
	 * 
	 * @param key EMF registry key
	 * @return
	 */
	protected boolean isQImpressAdapter(Collection<?> key) {
		for (Object o : key) {			
			if (o instanceof String) {
				String s = (String) o;
				// this code is little bit nasty but it is the simplest way of filtering just q-impress packages
				// TODO provide an extension point for selecting interesting packages				
				if (s.contains("q-impress") 
						|| s.contains("qimpress")
						|| s.contains("fzi")
						|| s.contains("uka.de")
						|| s.contains("ecore")
					) {
					return true;
				}
			}
		}
		
		return false;		
	}
	
	public synchronized void initializeFactoryProvider() {
		// the method can be called from different threads, so we need to ensure unique lazy initialization  
		
		if (adapterFactory == null) {
				logger.info("Initializing adapter factory for the accessible Q-ImPrESS models ...");
				adapterFactory = new ComposedAdapterFactory(createFactoryList());				
				logger.info("Initialization finished");
		}
	}
		
	public final ComposedAdapterFactory getAdapterFactory() {
		initializeFactoryProvider();
		
		return adapterFactory;
	}

	public void setOnlyQImpressFactories(boolean onlyQImpressFactories) {
		this.onlyQImpressFactories = onlyQImpressFactories;
	}

	public boolean isOnlyQImpressFactories() {
		return onlyQImpressFactories;
	}	
}
