/**
 * 
 */
package eu.qimpress.analysis.jpfcheck;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;

import org.eclipse.core.runtime.FileLocator;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.ui.console.MessageConsoleStream;
import org.osgi.framework.Bundle;

import eu.qimpress.analysis.core.AnalysisTask;
import eu.qimpress.analysis.generic.GenericAnalysisEObjectParam;
import eu.qimpress.analysis.generic.GenericAnalysisParam;
import eu.qimpress.analysis.generic.GenericAnalysisStringParam;
import eu.qimpress.analysis.generic.GenericAnalysisTask;
import eu.qimpress.ide.analysis.IAnalysisTool;
import eu.qimpress.samm.staticstructure.ComponentType;
import eu.qimpress.samm.staticstructure.InterfacePort;

/**
 * Analysis support tool based on JPF.
 * 
 * The implementation takes model of analysis and adapt its parameters to be passed to JPFcheck tool.
 * 
 * FIXME: this implementation is just a proof-of-concept and needs major refactoring
 * 
 * @author Michal Malohlava
 *
 */
public class JPFCheckAnalysisTool implements IAnalysisTool {
	
	/* Name of a temporary directory used by JPF to generate artifacts - stored in plugin's state store */ 
	protected static final String TMP_DIRECTORY = "tmp";
	/* Name of directory containing bundle libraries */
	protected static final String LIB_DIRECTORY = "lib/jpfcheck-bp";
	/* Names of jar files which have to be passed to JPF directly */ 
	protected static final String[] INTERNAL_JARS = new String[] { "jpfchecker-model.jar" , "env_jpf.jar" };
	/* Names of jar file which are needed for JPFCheck internal compilation task */
	protected static final String[] LIBRARIES_JARS = new String[] { "bcel.jar", 
		"behprotocols.jar", "evn_jvm.jar", "env_jpf.jar", "jpf.jar", "jpfchecker-model.jar", "jpfchecker-tool.jar", "JSAP-2.0.jar"  };
	
	

	@Override
	public void launch(AnalysisTask task, MessageConsoleStream msgStream) {
		
		EList<GenericAnalysisParam> analysisParams = ((GenericAnalysisTask) task).getParams();
		
		/* component can be primitive or composite, but both of them inherit from ComponentType */
		ComponentType componentType = (ComponentType) getEObjectParam("component", analysisParams);
		/* Component name */
		String componentName = componentType.getName();
		/* FIXME Component implementation ! It is not in SAMM metamodel?! */
		String componentImpl = getStringParam("componentImpl", analysisParams);
		
		/* Provided interfaces */
		Map<String, String> provIfaces = new HashMap<String, String>();
		for(InterfacePort p : componentType.getProvided()) {
			provIfaces.put(p.getName(), getStringParam(p.getInterfaceType().getName() + "Impl", analysisParams));
		}
		/* Required interfaces */
		Map<String, String> reqIfaces = new HashMap<String, String>();
		for(InterfacePort p : componentType.getRequired()) {
			reqIfaces.put(p.getName(), getStringParam(p.getInterfaceType().getName() + "Impl", analysisParams));
		}
		
		/* JPF settings */
		File protocolFile = getProtocolFile(analysisParams);		
		String valueSetsClassName = getStringParam("valueSetsClassName", analysisParams);
		/* Temporary directory */
		File tempDir = getTemporaryDir();
		/* References to internal jars required by JPF */ 
		Collection<File> internalJarFiles = getInternalJarFiles(INTERNAL_JARS);
		internalJarFiles.add(tempDir);
		/* References to application binaries stored in workspace directory */
		Collection<File> appBinaries = getApplicationJarFiles(analysisParams);
		
		/* DEBUG output showing analysis parameters */
		printAnalysisParameters(
				componentName, 
				componentImpl,
				provIfaces, 
				reqIfaces, 
				protocolFile,
				valueSetsClassName,
				tempDir,
				internalJarFiles,
				msgStream);
		/* --- END of Debug output */
		
		/* ----------------------------------- */
		Thread currentThread = Thread.currentThread();
		ClassLoader oldClassloader = currentThread.getContextClassLoader();
		try {
			Collection<URL> urls = new LinkedList<URL>();
			for (File f : internalJarFiles) {
				urls.add(f.toURI().toURL());
			}
			for(File f: appBinaries) {
				urls.add(f.toURI().toURL());
			}
			urls.add(tempDir.toURI().toURL());
			
			ClassLoader newClassLoader = new URLClassLoader(urls.toArray(new URL[] {}));
			currentThread.setContextClassLoader(newClassLoader);
			
		} catch (Exception e) {
			e.printStackTrace();
		}
		
		
		/* ----------------------------------- */
		// prepare class path for JPFChecker compilation
		StringBuffer sb = new StringBuffer(System.getProperty("java.class.path"));
		for (File f : getInternalJarFiles(LIBRARIES_JARS)) {
			sb.append(File.pathSeparatorChar);
			sb.append(f.getAbsolutePath());
		}
		for (File f : appBinaries) {
			sb.append(File.pathSeparatorChar);
			sb.append(f.getAbsolutePath());			
		}
		
		System.setProperty("java.class.path", sb.toString());
		try {
			/* call PP's checker tool */				
			checker.Main.runFromEclipse(
					componentName, 
					componentImpl,
					provIfaces, 
					reqIfaces, 
					protocolFile,
					valueSetsClassName,
					tempDir,
					internalJarFiles,
					msgStream);
		} finally {
			/* ----------------------------------- */
			currentThread.setContextClassLoader(oldClassloader);
			/* ----------------------------------- */
		}
	}

	/**
	 * Return reference to temporary directory.
	 * 
	 * @return temporary directory
	 */
	protected File getTemporaryDir() {
		File tempDir = JPFCheckPlugin.getDefault().getStateLocation().append(TMP_DIRECTORY).toFile();
		return tempDir;		
	}	
	
	/**
	 * Return a collection of references to internal jars stored in bundle.
	 * 
	 * Note: bundle has to be deployed unpacked to received concrete files!
	 * 
	 * @param jarNames names of jar which are required
	 * 
	 * @return collection of files representing bundles jars.
	 */
	protected Collection<File> getInternalJarFiles(String[] jarNames) {
		Bundle thisBundle = JPFCheckPlugin.getDefault().getBundle();
		Collection<File> result = new LinkedList<File>();
		
		for (String jarName : jarNames) {
						
			try {
				URL url = thisBundle.getEntry("/" + LIB_DIRECTORY + "/" +  jarName);
				if (url != null) {
					result.add(new File(FileLocator.toFileURL(url).toURI()));
				}
			} catch (IOException e) {				
				e.printStackTrace();
			} catch (URISyntaxException e) {
				e.printStackTrace();
			}
		}
				
		return result;
	}
	
	protected Collection<File> getApplicationJarFiles(EList<GenericAnalysisParam> params) {
		Collection<File> result = new LinkedList<File>();
		
		String appBinariesDir = getStringParam("appBinariesDir", params);
		if (appBinariesDir != null) {
			URL appBinDirURL;
			
			try {
				appBinDirURL = FileLocator.toFileURL(new URL(appBinariesDir));
				File appBinDir = new File(appBinDirURL.toURI());
				if (appBinDir.exists() && appBinDir.isDirectory()) {
					File[] jarFiles = appBinDir.listFiles(new FilenameFilter() {
						
						@Override
						public boolean accept(File dir, String name) {
							if (name.endsWith(".jar")) 
								return true;
							else
								return false;
						}
					});	
					for (File f : jarFiles) {
						result.add(f);
					}
				}
				
			} catch (MalformedURLException e) {				
				e.printStackTrace();
			} catch (IOException e) {				
				e.printStackTrace();
			} catch (URISyntaxException e) {
				e.printStackTrace();
			}
		}
		
		
		
		return result;
	}
	
	protected File getProtocolFile(EList<GenericAnalysisParam> params) {
		File result = null;
		
		String protocolFile = getStringParam("protocolFile", params);
		if (protocolFile != null) {
			URL protocolFileURL;
			try {
				protocolFileURL = FileLocator.toFileURL(new URL(protocolFile));
				if (protocolFileURL != null) {
					result = new File(protocolFileURL.toURI());
				}
			} catch (MalformedURLException e) {
				e.printStackTrace();
			} catch (IOException e) {
				e.printStackTrace();
			} catch (URISyntaxException e) {
				e.printStackTrace();
			}
		}
		
		return result;
	}
	
	protected void printAnalysisParameters(String componentName, String componentImpl, Map<String, String> provIfaces, Map<String, String> reqIfaces, File protocolFile, String valueSetsClassName, File tempDir, Collection<File> internalJarFiles, MessageConsoleStream msgStream) {
		msgStream.println("JPFCheck tool");
		msgStream.println("------------------");
		msgStream.println("Input parameters:");
		msgStream.println("  componentName: " + componentName);
		msgStream.println("  componentImpl: " + componentImpl);
		msgStream.println("  provided ifaces:");
		for(Map.Entry<String, String> e : provIfaces.entrySet()) {
			msgStream.println("    " + e.getKey() + " : " + e.getValue());									
		}
		msgStream.println("  required ifaces:");
		for(Map.Entry<String, String> e : reqIfaces.entrySet()) {
			msgStream.println("    " + e.getKey() + " : " + e.getValue());									
		}
		msgStream.println("  protocolFile: " + protocolFile);
		msgStream.println("  valueSetsClassName: " + valueSetsClassName);
		msgStream.println("  tempDir: " + tempDir);
		msgStream.println("  internal JARS:" );
		for(File f : internalJarFiles) {
			msgStream.println("    " + f);
		}
		msgStream.println("------------------");
	}
	
	private String getStringParam(String name, EList<GenericAnalysisParam> params) {
		String result = null;
		for (GenericAnalysisParam param : params) {
			if (param instanceof GenericAnalysisStringParam) {
				GenericAnalysisStringParam sParam = (GenericAnalysisStringParam) param;
				if (sParam.getName().equals(name)) {
					return sParam.getValue();
				}
			}
		}
		
		return result;
	}
	
	private EObject getEObjectParam(String name, EList<GenericAnalysisParam> params) {
		EObject result = null;
		for (GenericAnalysisParam param : params) {
			if (param instanceof GenericAnalysisEObjectParam) {
				GenericAnalysisEObjectParam sParam = (GenericAnalysisEObjectParam) param;
				if (sParam.getName().equals(name)) {
					return sParam.getValue();
				}
			}
		}
		
		return result;
	}
	
}
