
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <sys/times.h>

#include <iostream>

using namespace std;

static const unsigned long long NANOSECONDS_IN_SEC = 1000000000ULL;

#define print_clock_res(res, clk_id) 			\
	if (clock_getres(clk_id, &res) != 0) {		\
		perror("clock_getres(" #clk_id ")");	\
	} else {									\
		cout << #clk_id << " resolution is "	\
		<< res.tv_sec << "s " 					\
		<< res.tv_nsec << "ns" << endl;			\
	}

void test_relative_sleep(clockid_t clk_id, unsigned long long nsSingle, unsigned long long nsTotal) {
	struct tms tmsStart, tmsEnd;

	unsigned cntFull = nsTotal / nsSingle;
	unsigned long long nsRemain = nsTotal % nsSingle;

	struct timespec tsSingle, tsRemain;

	tsSingle.tv_sec = nsSingle / NANOSECONDS_IN_SEC;
	tsSingle.tv_nsec = nsSingle % NANOSECONDS_IN_SEC;

	if (nsRemain != 0) {
		tsRemain.tv_sec = nsRemain / NANOSECONDS_IN_SEC;
		tsRemain.tv_nsec = nsRemain % NANOSECONDS_IN_SEC;
	}

	cout << "total sleep of " << nsTotal << "ns done in " << cntFull << " sleeps of " << nsSingle << "ns and remaining " << nsRemain << "ns" << endl;

	clock_t realStart = times(&tmsStart);
	for (unsigned long long i = 0; i < cntFull; i++) {
		clock_nanosleep(clk_id, 0, &tsSingle, NULL);
	}
	if (nsRemain != 0) {
		clock_nanosleep(clk_id, 0, &tsRemain, NULL);
	}

	clock_t realEnd = times(&tmsEnd);

	cout << "real: " << realEnd - realStart << endl;
	cout << "user: " << tmsEnd.tms_utime - tmsStart.tms_utime << endl;
	cout << "syst: " << tmsEnd.tms_stime - tmsStart.tms_stime << endl;
}

void test_absolute_sleep(clockid_t clk_id, unsigned long long nsSingle, unsigned long long nsTotal) {
	struct tms tmsStart, tmsEnd;

	struct timespec tsSingle, tsAbsDeadline;

	tsSingle.tv_sec = nsSingle / NANOSECONDS_IN_SEC;
	tsSingle.tv_nsec = nsSingle % NANOSECONDS_IN_SEC;

	clock_gettime(clk_id, &tsAbsDeadline);
	// this should not overflow
	tsAbsDeadline.tv_sec += nsTotal / NANOSECONDS_IN_SEC;
	const unsigned long nsec = tsAbsDeadline.tv_nsec + nsTotal % NANOSECONDS_IN_SEC;
	if (nsec > NANOSECONDS_IN_SEC) {
		tsAbsDeadline.tv_nsec = nsec % NANOSECONDS_IN_SEC;
		tsAbsDeadline.tv_sec += 1;
	} else {
		tsAbsDeadline.tv_nsec = nsec;
	}

	cout << "total sleep of " << nsTotal << "ns done in sleeps of " << nsSingle << "ns" << endl;

	clock_t realStart = times(&tmsStart);
	struct timespec tsAbs;
	unsigned long long cycles = 0;
	while (true) {
		clock_gettime(clk_id, &tsAbs);

		// calculate new absolute timer expiration time
		tsAbs.tv_sec += tsSingle.tv_sec;
		// this should not overflow, unsigned long can handle
		const unsigned long nsec = tsAbs.tv_nsec + tsSingle.tv_nsec;
		if (nsec > NANOSECONDS_IN_SEC) {
			tsAbs.tv_nsec = nsec % NANOSECONDS_IN_SEC;
			tsAbs.tv_sec += 1;
		} else {
			tsAbs.tv_nsec = nsec;
		}
		// should this be the last sleep?
		if (tsAbs.tv_sec > tsAbsDeadline.tv_sec || (tsAbs.tv_sec == tsAbsDeadline.tv_sec && tsAbs.tv_nsec >= tsAbsDeadline.tv_nsec)) {
			clock_nanosleep(clk_id, TIMER_ABSTIME, &tsAbsDeadline, NULL);
			break;
		}
		clock_nanosleep(clk_id, TIMER_ABSTIME, &tsAbs, NULL);
		cycles++;
	}

	clock_t realEnd = times(&tmsEnd);

	cout << "performed " << cycles << " cycles which is " << nsTotal/cycles << "ns per cycle" << endl;

	cout << "real: " << realEnd - realStart << endl;
	cout << "user: " << tmsEnd.tms_utime - tmsStart.tms_utime << endl;
	cout << "syst: " << tmsEnd.tms_stime - tmsStart.tms_stime << endl;
}

int main(int argc, char * argv[]) {
	struct timespec res;
	cout << "Supported clocks and detected resolutions: " << endl;
	print_clock_res(res, CLOCK_MONOTONIC);
	print_clock_res(res, CLOCK_REALTIME);
	print_clock_res(res, CLOCK_PROCESS_CPUTIME_ID);
	print_clock_res(res, CLOCK_THREAD_CPUTIME_ID);
	cout << endl;

	if (argc != 5) {
		cerr << "Wrong nunmber of arguments" << endl;
		cout << "usage: " << argv[0] << " CLOCK_ID {abs,rel} total_time_in_seconds time_per_sleep_in_nanoseconds" << endl;
		cout << "Program will run clock_nanosleep using a specified timer, absolute/relative waiting, with given" << endl;
		cout << "time per one sleep and total time to sleep" << endl;
		cout << "Note that with relative waits, the overhead accumulates and short times per sleep may cause it to run for a long time!" << endl;
		exit(EXIT_FAILURE);
	}

	// parse arguments
	string clk = argv[1];
	clockid_t clk_id;
	if (clk == "CLOCK_MONOTONIC") {
		clk_id = CLOCK_MONOTONIC;
	} else if (clk == "CLOCK_REALTIME") {
		clk_id = CLOCK_REALTIME;
	} else if (clk == "CLOCK_PROCESS_CPUTIME_ID") {
		clk_id = CLOCK_PROCESS_CPUTIME_ID;
	} else if (clk == "CLOCK_THREAD_CPUTIME_ID") {
		clk_id = CLOCK_THREAD_CPUTIME_ID;
	} else {
		cerr << "unknown clock parameter: " << clk << endl;
		exit(EXIT_FAILURE);
	}

	string sAbsRel = argv[2];
	bool absolute;
	if (sAbsRel == "abs") {
		absolute = true;
	} else if (sAbsRel == "rel") {
		absolute = false;
	} else {
		cerr << "unknown abs/rel parameter: " << sAbsRel << endl;
		exit(EXIT_FAILURE);
	}

	unsigned long long int sTotal = strtoull(argv[3], NULL, 10);
	unsigned long long int nsSingle = strtoull(argv[4], NULL, 10);

	if (absolute) {
		test_absolute_sleep(clk_id, nsSingle, sTotal*NANOSECONDS_IN_SEC);
	} else {
		test_relative_sleep(clk_id, nsSingle, sTotal*NANOSECONDS_IN_SEC);
	}
}

