package rpg;

import static rpg.ConfigReader.GROUP_APP;
import static rpg.ConfigReader.OUTPUT_CONTEXT_CONFIG;
import static rpg.ConfigReader.OUTPUT_CONTEXT_SHARED;

import java.util.Iterator;
import java.util.PriorityQueue;
import java.util.Collections;
import java.util.Vector;
import java.util.concurrent.BrokenBarrierException;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Semaphore;
import java.util.concurrent.CyclicBarrier;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.io.PrintStream;

/**
 * A manager organizing multiple executions of Workers. A Worker performs
 * one execution and measurements of the module tree. This class calls several
 * workers concurrently simulating a server under stress from clients, thus
 * providing the measurements with background workload, so they are not just
 * measured in isolation.
 */
public class WorkerManager {
	/**
	 * Workers managed by this manager - clients that calls this server.
	 */
	private Worker[] workers;

	/**
	 * Maximum number of threads this server has available for client requests.
	 */
	private int threadCount;

	/**
	 * The number of simulated clients "calling" the manager to performs their
	 * execution. This is the number of client request times at a time, in the
	 * clientTimes queue.
	 */
	private int clients;
	/**
	 * A parameter of an exponential distribution for the random times between
	 * client requests.
	 */
	private int clientTimeEx;

	/** Global lock for Worker threads.
	 *
	 * Used for module initialization, deinitialization and isolated measurements. 
	 */
	private final Lock _exclusiveLock = new ReentrantLock ();
	
	public final void exclusiveLock () {
		_exclusiveLock.lock ();
	}
	
	public final void exclusiveUnlock () {
		_exclusiveLock.unlock ();
	}
	
	/** Global barrier for synchronizing most phases of Worker threads. */
	private CyclicBarrier workerBarrier;
	
	/** Barrier wait method for most phases of Workers.
	 *  
	 * @throws BrokenBarrierException 
	 * @throws InterruptedException
	 */
	public final int waitBarrier () {
		/* We will not process barrier exceptions in any special way */
		try {
			return workerBarrier.await ();
		} catch (InterruptedException e) {
			throw new RuntimeException (e);
		} catch (BrokenBarrierException e) {
			throw new RuntimeException (e);
		}
	}
	
	/** Global barrier for synchronizing the phase of Workers before starting shared measurements.
	 *  
	 * This barrier additionally creates initial client requests and start measuring client time.
	 * @see generateClientsRequests()
	 */
	private CyclicBarrier requestsGenBarrier;

	/** Barrier wait method for the phase of Workers before starting shared measurements.
	 *  
	 * @throws BrokenBarrierException 
	 * @throws InterruptedException
	 */
	public final int waitRequestsGenBarrier () {
		/* We will not process barrier exceptions in any special way */
		try {
			return requestsGenBarrier.await ();
		} catch (InterruptedException e) {
			throw new RuntimeException (e);
		} catch (BrokenBarrierException e) {
			throw new RuntimeException (e);
		}
	}
	
	/**
	 * A queue of scheduled client time requests.
	 */
	private PriorityQueue<Long> clientTimes = new PriorityQueue<Long> ();

	/** Timestamp of the moment where isolated measurements end
	 * and client processing starts
	 */
	private long clientStartTime;
	
	/** Tracks the number of request cycles remaining to achieve the user-set minimum. */
	private CountDownLatch remainingCycles;

	/** Update cycle count and notify the main thread when requested minimum was achieved. */
	public final void addCycle () {
		remainingCycles.countDown ();
	}
	
	/** Generates a single client request time - schedules the client request.
	 * 
	 * This method should be called in synchronized (clientTimes) block.
	 */
	private final void generateClientRequest () {
		final long thinkTime = (long) (Main.NS_IN_US * (Rand.randExp (clientTimeEx))); 
		final long clientTime = System.nanoTime () + thinkTime;
		clientTimes.add (clientTime);
	}
	
	/** Gets the earliest client request from the priority queue. 
	 *
	 * This method should be called in synchronized (clientTimes) block.
	 */
	private final long getClientRequest () {

		if (!clientTimes.isEmpty ()) {
			/* take the client that's on top of the queue and return its time */
			return clientTimes.poll ();
		} else {
			// if the queue is empty, it means something's seriously wrong
			throw new AssertionError ("clientTimes queue is empty");
		}
	}
	
	/** One worker thread calls this to generate the first bunch of clients
	 * and denote start of non-isolated work.
	 * 
	 * Declared as Runnable instead of a function, to be used by
	 * the CyclicBarrier requestsGenBarrier
	 */
	Runnable startClientRequests = new Runnable() {
		public void run() {
			synchronized (clientTimes) {
				for (int i = 0; i < clients; i++) {
					generateClientRequest();
				}
			}
			
			/* get timestamp for main thread's accounting */
			clientStartTime = System.nanoTime ();
		}
	};

	/** Worker threads call this to finish serving one client and get a new one */
	long getNextClientRequest () {
		synchronized (clientTimes) {

			/* Generate new time for the client that was just served */
			generateClientRequest ();

			/* Get the client with earliest due time */
			return getClientRequest();
		}
	}
	
	/** Worker thread calls this to get its first client */
	long getFirstClientRequest () {
		synchronized (clientTimes) {
			/* Get the client with earliest due time */
			return getClientRequest();
		}
	}	

	/**
	 * Start the manager - server.
	 */
	void start (int threadCount, int clientWaitTime, int clientCount, int minCycles) {
		ModuleBase.rpgTerminate = false;

		this.threadCount = threadCount;
		clients = clientCount;
		clientTimeEx = clientWaitTime;

		/* the performance model needs to know this */
		ConfigReader.output (GROUP_APP, OUTPUT_CONTEXT_CONFIG, ConfigReader.ITEM_CLIENT_COUNT, clientCount);
		ConfigReader.output (GROUP_APP, OUTPUT_CONTEXT_CONFIG, ConfigReader.ITEM_THREAD_COUNT, threadCount);
		ConfigReader.output (GROUP_APP, OUTPUT_CONTEXT_CONFIG, ConfigReader.ITEM_CLIENT_WAIT_TIME_EX, clientTimeEx);

		workerBarrier = new CyclicBarrier (threadCount);
		requestsGenBarrier = new CyclicBarrier (threadCount, startClientRequests);
		remainingCycles = new CountDownLatch (minCycles);
		
		/* create threads and start them */
		workers = new Worker[threadCount];
		for (int i = 0; i < threadCount; i++) {
			workers[i] = new Worker(this);
			workers[i].start();
		}
	}

	/** Wait for the workers to finish, or timeout on max_time. 
	 *
	 * @throws InterruptedException from remainingCycles.await() and Thread.sleep()
	 */
	public void waitForWorkers(int minTime, int maxTime) throws InterruptedException {
		/* Now we need to determine when to end the application.
		 * First we get the current time as the start time.
		 */
		final long startTime = System.nanoTime ();
		
		/* First we wait until there are enough cycles collected.
		 * We have a CountDownLatch that the workers decrease on each finished client request.
		 * If there is a maxTime limit, we use it as a timeout.
		 */
		if (maxTime != 0) {
			if (!remainingCycles.await (maxTime, TimeUnit.SECONDS)) {
				System.err.println ("Application timeout when waiting for enough cycles");
				ModuleBase.rpgTerminate = true;
			}
		} else {
			remainingCycles.await ();
		}

		/* If there wasn't a timeout already */
		if (!ModuleBase.rpgTerminate) {
			/*
			 * We collected enough samples, now we should wait to achieve minimal time run in the shared mode,
			 * which should be counted from the clientStartTime (excluding initialization and isolated measurement)
			 * But the maxTime is counted from the startTime timestamp, so we need to see which
			 * deadline is earlier
			 */
			/* minTime and maxTime is in seconds */
			final long maxTimeDeadline = startTime + TimeUnit.SECONDS.toNanos (maxTime);
			final long clientTimeDeadline = clientStartTime + TimeUnit.SECONDS.toNanos (minTime);
			long finalDeadLine;
			if (maxTime != 0 && maxTimeDeadline < clientTimeDeadline) {
				/* maxTime is earlier */
				System.err.println ("Application timeout is earlier than min_time");
				finalDeadLine = maxTimeDeadline;
			} else {
				finalDeadLine = clientTimeDeadline;
			}

			/* Is there even a reason to wait? If yes, wait... */
			final long sleepTime = finalDeadLine - System.nanoTime();
			if (sleepTime > 0) {
				final long sleepTimeMs = sleepTime / Main.NS_IN_MS;
				final int sleepTimeNs = (int) (sleepTime % Main.NS_IN_MS);
				Thread.sleep(sleepTimeMs, sleepTimeNs);
			}
		}

		/* Either way, time to end the workers */
	}
	
	/** Set the termination flag and wait for worker threads to join.
	 * 
	 * @throws InterruptedException from Thread.join ()
	 */
	boolean terminate (boolean wait) throws InterruptedException {
		ModuleBase.rpgTerminate = true;

		if (wait) {
            /* TODO: configurable variable? */
			long milisecondsToWait = TimeUnit.SECONDS.toMillis (120);

			for (int i = 0; i < threadCount; i++) {
				workers[i].join (milisecondsToWait);
				// the join () doesn't distinguish success and timeout
				if (workers[i].isAlive ()) {
					return false;
				}
			}
		} else {
			for (int i = 0; i < threadCount; i++) {
				workers[i].join();
			}
		}
		return true;
	}

	/**
	 * Prints all the measured values.
	 */
	void printTimes () {
		for (int i = 0; i < threadCount; i++) {
			workers[i].printTimes ();
		}
	}
}