package rpg;


import static rpg.DummyMacros.*;
import static rpg.ModuleBase.rpgTerminate;

/* This is needed for the printtimes macro */  
import static rpg.ConfigReader.GROUP_APP;
import static rpg.ConfigReader.OUTPUT_CONTEXT_SHARED;

/** This is to catch the use of thread context classloader in modules, which could potentially break the module isolation */
class DummyClasloader extends ClassLoader {
	@Override
	protected synchronized Class<?> loadClass(String name, boolean resolve)
			throws ClassNotFoundException {
		throw new RuntimeException ("ContextClassLoader should not be used by modules");
	}
}

/**
 * A wrapper for running a single instance of a module tree. Simulates a client
 * request, while the WorkerManager simulates the server managing the requests.
 * Each worker has its own thread where it runs its workload.
 */
public class Worker extends Thread {
	/** Identifier of the WorkerManager handling this worker. */
	private WorkerManager owner;

	/** An array for storing processing times
	 * 
	 * Three values per client request are stored:
	 * - monotonic timestamp when the work starts
	 * - monotonic timestamp when work is finished
	 * - monotonic timestamp of client arrival
	 */
	private long [] processingTimes;

	/** Counter of stored times (triplets). */
	private int storedTimes;
	
	/** How many values are stored per client request. */
	private static int valuesPerClient = 3;
	
	/** Stores the arrival time of the currently processed client request. */
	private long currentClientArrival;

	/* 
	 * This macro is expanded to loading module classes via unique classloaders - one for each module.
	 * The prefix before the macro itself is expected by the generator. We need to have it here
	 * for Eclipse editing, to yield correct Java syntax without preprocessing. 
	 */
	private static final Class RPG_EXPAND_DEFINE = null;
	
	/** Check if client_time is due and wait if necessary
	 *  
	 * @throws InterruptedException from Thread.sleep ()
	 */
	private void waitForClientRequest () throws InterruptedException {
		final long currentTime = System.nanoTime();
		final long sleepTime = currentClientArrival - currentTime; 

		/* For clients that already expired, we can immediately serve them */
		if (sleepTime > 0) {
			/* For clients not yet expired, we just wait for their due time.
			 * Note that it is not a problem if another client appears with
			 * a shorter due time than this one, because this only happens
			 * when the worker has finished the client and generated its new
			 * time, so the worker is also free to process it.
			 */
			final long sleepTimeMs = sleepTime / Main.NS_IN_MS;
			final int sleepTimeNs = (int) (sleepTime % Main.NS_IN_MS);
			Thread.sleep(sleepTimeMs, sleepTimeNs);
		}
	}
	
	/** Print request processing times collected during work */
	public void printTimes() {
		final String timePrefix = ConfigReader.outputCreatePrefix 
			(GROUP_APP, OUTPUT_CONTEXT_SHARED, ConfigReader.OUTPUT_MONOTONIC_MEASURE);
		final String endPrefix = ConfigReader.outputCreatePrefix 
			(GROUP_APP, OUTPUT_CONTEXT_SHARED, ConfigReader.OUTPUT_MONOTONIC_MEASURE + "_end");
		final String clientPrefix = ConfigReader.outputCreatePrefix 
			(GROUP_APP, OUTPUT_CONTEXT_SHARED, ConfigReader.OUTPUT_MONOTONIC_MEASURE + "_client");
		for (int i = 0; i < storedTimes; i++) {
			
			final int index = i * valuesPerClient;
			
			final long start = processingTimes [index + 0];
			final long end = processingTimes [index + 1];
			final long client = processingTimes [index + 2];
			
			ConfigReader.outputWithPrefix (timePrefix, end - start);
			ConfigReader.outputWithPrefix (endPrefix, end);
			ConfigReader.outputWithPrefix (clientPrefix, end - client);
		}

	}

	/**
	 * A constructor.
	 * @param owner The WorkerManager managing this worker.
	 */
	Worker (WorkerManager owner) {
		this.owner = owner;
		
		/* Disable usage of getContextClassloader. */
		this.setContextClassLoader (new DummyClasloader());
		
		/* Allocate the array for storing results. */
		processingTimes = new long [ModuleTimed.getMaxMeasuredValues() * valuesPerClient];
	}

	/**
	 * An equivalent of the do_work() function in the original C++ template
	 * file.
	 * It is called by the system when the thread of this worker is started.
	 * Calls the different stages of the workload. This is the method that needs
	 * filling by the random program generator, that supplies the modules, that
	 * is - the workload to run.
	 */
	@Override
	public void run() {

		/* Declare and initialize individual module instances. The initialization is made under the
		 * exclusive lock to cover cases where legacy modules access static variables not protected
		 * by locks.
		 */
		owner.exclusiveLock ();
		/* It would be better to have this in the try-finally block as well, but that would require splitting
		 * the declaration and creation of objects in two functions.
		 */
		RPG_EXPAND_DECLARE();
		try {
			RPG_EXPAND_INIT();
		} finally {
			owner.exclusiveUnlock ();
		}

		/* Before the isolated measurement, print implementation-specific module configuration for
		 * the module transformation purposes. Only one thread prints this as module instances
		 * in each thread are the same.
		 */
		if (owner.waitBarrier () == 0) {
			RPG_EXPAND_PRINTCONFIG();
		}

		/* The chosen thread should finish printing before isolated measurements. */
		owner.waitBarrier ();
		
		/* Perform isolated measurements. The code that drives the isolated measurements
		 * is, somewhat unusually, inside the modules, because expanding it here would
		 * make the generator a bit more complex.
		 * 
		 * The isolation is achieved by the exclusive mutex.
		 */
		owner.exclusiveLock ();
		try {
			RPG_EXPAND_MEASURE();
		} finally {
			owner.exclusiveUnlock ();
		}
		
		/* Wait for the isolated measurements to be finished. */
		owner.waitBarrier (); 

		/* Reset the collected measurements before the shared work commences. */
		RPG_EXPAND_CLEAR();
		
		/*
		 * Wait for the reset to be finished. Then generate the initial client requests
		 * in the queue and start measuring the client time. This is done by using
		 * a special barrier which performs an action when all threads arrive and
		 * before any thread is released.
		 */
		owner.waitRequestsGenBarrier();

		/* Get the first client */
		currentClientArrival = owner.getFirstClientRequest ();
		
		/* Process clients until we have to terminate */
		while (!rpgTerminate) {
			try {
				/* Wait for the client if needed. */
				waitForClientRequest ();
				work (
						RPG_EXPAND_WORK
				);
			} catch (Exception e) {
				e.printStackTrace ();
				return;
			}

			/* Generate and get the next client. */
			currentClientArrival = owner.getNextClientRequest ();
		}

		/* We know that we should terminate, but not all threads get the information
		 * at the same time. Wait for them, the measurements are no longer collected
		 * so it does not matter that we do not generate the workload.
		 */
		owner.waitBarrier ();
		
		/* The OUTPUT_CONTEXT_SHARED is used here to ensure the import is not removed
		 * due to unused import warning. The generated calls rely on it it.
		 */
		RPG_EXPAND_PRINTTIMES (ConfigReader.OUTPUT_CONTEXT_SHARED);

		/* Wait for all threads to finish printing */
		owner.waitBarrier ();
	}

	/**
	 * Executes the workload of the module. Specified module should be the root
	 * module of the module tree, which in turn executes its submodules.
	 * @param m the Module to execute
	 */
	void work(Module m) {
	    /* The time is collected both here and inside the module. Here, absolute
	     * times are recorded (to make it possible to correctly calculate
	     * client perceived timing behavior).
	     */
		long startTime = System.nanoTime();
		SessionStateHolder sessionState = new SessionStateHolder (SessionState.RETURN_OK);
		m.measuredWork (sessionState);
		long endTime = System.nanoTime();

		/* Time is not collected when terminating, some threads might have already finished. */
		if (!rpgTerminate && storedTimes < ModuleTimed.getMaxMeasuredValues()) {
			final int index = storedTimes * valuesPerClient;
			
			processingTimes [index + 0] = startTime;
			processingTimes [index + 1] = endTime;
			processingTimes [index + 2] = currentClientArrival;
			
			storedTimes ++;
		}

		// Update the cycle count and notify the main thread when the requested minimum was achieved.
		owner.addCycle ();
	}

}
