#include <time.h>
#include <sys/time.h>
#include <stdio.h>
#include <stdlib.h>
#include <iostream>
#include <pthread.h>
#include <vector>
#include <memory>
#include <semaphore.h>
#include <queue>
#include <math.h>

#include "common.h"
#include "config.h"
#include "parser.h"

using namespace std;
using namespace rpg;

#include "rpg_ConfigReader.h"

/*
 * This file contains the native implemementation of the JNI interface defined
 * in the ConfigReader.java class. *
 */

void ThrowIllegalArgExc(JNIEnv *env, const char *msg);
rpg_data_source_t retrieveValue(JNIEnv *env, jstring group, jstring item);


JNIEXPORT void JNICALL Java_rpg_ConfigReader_loadConfig
	(JNIEnv *env, jclass caller_class, jstring config_file_name) {

	// Convert java 16 bit filename string to 8-bit style.
	const char *cpp_file_name =
		env->GetStringUTFChars(config_file_name, NULL);

	if(cpp_file_name == NULL) {
		ThrowIllegalArgExc(env, "Cannot work with the configuration file name.");
		return;
	}

	/* first, parse default configuration file, if it exists */
	FILE* input = fopen(cpp_file_name, "r");
	if (input == NULL) {
		ThrowIllegalArgExc(env, "Cannot read the configuration file.");
		return;
	}
	
	rpg_context_t context(sprovider, cprovider);

	/* try to parse configuration file */
	config_set_in(input);
	if (config_parse(context) != 0 || errors != 0) {
		fclose(input);
		ThrowIllegalArgExc(env, "Error parsing the configuration file.");
		return;
	}
	
	fclose(input);
	/* parse done! */
	
	// Free 8-bit version of the string.
	env->ReleaseStringUTFChars(config_file_name, cpp_file_name);

	/* free lex structures */
	config_lex_destroy();
}

JNIEXPORT jint JNICALL Java_rpg_ConfigReader_getIntItem
	(JNIEnv *env, jclass caller_class, jstring group, jstring item) {

	auto_ptr<dataSource> value = retrieveValue(env, group, item);

	if(!value.get()) {
		ThrowIllegalArgExc(env, "Cannot retrive the data source.");
	}
	if(!value->isInteger()) {
		ThrowIllegalArgExc(env, "Not an integer data source.");
	}
	return value->getIntValue();
}

JNIEXPORT jstring JNICALL Java_rpg_ConfigReader_getStringItem
	(JNIEnv *env, jclass caller_class, jstring group, jstring item) {

	rpg_data_source_t value = retrieveValue(env, group, item);

	if(!value.get()) {
		ThrowIllegalArgExc(env, "Cannot retrive the data source.");
	}
	if(!value->isString()) {
		ThrowIllegalArgExc(env, "Not a string data source.");
	}

	const rpg_str_t& ret_val = value->getStringValue();
	return env->NewStringUTF(ret_val.c_str());
}

JNIEXPORT jint JNICALL Java_rpg_ConfigReader_getIntListItemEntry
	(JNIEnv *env, jclass caller_class, jstring group, jstring item, jint index) {

	rpg_data_source_t value = retrieveValue(env, group, item);

	if(!value.get()) {
		ThrowIllegalArgExc(env, "Cannot retrive the data source.");
	}
	if(!value->isIntegerArray()) {
		ThrowIllegalArgExc(env, "Not an integer array data source.");
	}
	if(value->array_size() >= (rpg_uint_t)index) {
		ThrowIllegalArgExc(env,
				"Too big index, the list does not have that much entries.");
	}

	return (value->access_int_list())[index];
}

JNIEXPORT jstring JNICALL Java_rpg_ConfigReader_getStringListItemEntry
	(JNIEnv *env, jclass caller_class, jstring group, jstring item, jint index) {

	rpg_data_source_t value = retrieveValue(env, group, item);

	if(!value.get()) {
		ThrowIllegalArgExc(env, "Cannot retrive the data source.");
	}
	if(!value->isStringArray()) {
		ThrowIllegalArgExc(env, "Not a string array data source.");
	}
	if(value->array_size() >= (rpg_uint_t)index) {
		ThrowIllegalArgExc(env,
				"Too big index, the list does not have that much entries.");
	}

	const rpg_str_t& ret_val = (*value)[index];
	return env->NewStringUTF(ret_val.c_str());
}

JNIEXPORT jint JNICALL Java_rpg_ConfigReader_getListItemLength
	(JNIEnv *env, jclass caller_class, jstring group, jstring item) {

	rpg_data_source_t value = retrieveValue(env, group, item);

	if(!value.get()) {
		ThrowIllegalArgExc(env, "Cannot retrive the data source.");
	}
	if(!value->isIntegerArray() && !value->isStringArray()) {
		ThrowIllegalArgExc(env, "Not a string array data source.");
	}

	return value->array_size();
}

JNIEXPORT jboolean JNICALL Java_rpg_ConfigReader_isItemVoid
	(JNIEnv *env, jclass caller_class, jstring group, jstring item) {

	rpg_data_source_t value = retrieveValue(env, group, item);

	if(!value.get()) {
		ThrowIllegalArgExc(env, "Cannot retrive the data source.");
	}
	if(value->isVoid())
		return JNI_TRUE;
	else
		return JNI_FALSE;
}

/** A helper function that retrieves a dataSource value from the config file.
 * Contains universal code that would be the same in all the functions
 * retrieving various types of values.
 */
rpg_data_source_t retrieveValue(JNIEnv *env, jstring group, jstring item) {
	// Convert java 16 bit filename string to 8-bit style.
	const char *cpp_group = env->GetStringUTFChars(group, NULL);
	const char *cpp_item = env->GetStringUTFChars(item, NULL);

	rpg_data_source_t null_ptr;
	if(cpp_group == NULL) {
		ThrowIllegalArgExc(env, "Cannot convert a group string.");
		return null_ptr;
	}
	if(cpp_item == NULL) {
		env->ReleaseStringUTFChars(group, cpp_group);
		ThrowIllegalArgExc(env, "Cannot convert an item string.");
		return null_ptr;
	}
	// Now retrieve the value.
	rpg_data_source_t value(sprovider.createSource(cpp_group, cpp_item));

	env->ReleaseStringUTFChars(group, cpp_group);
	env->ReleaseStringUTFChars(item, cpp_item);

	return value;
}

/** Exception utility function derived from the one in JNI specification.
 */
void ThrowIllegalArgExc(JNIEnv *env, const char *msg) {
	cerr << msg << endl;
	jclass cls = env->FindClass("java/lang/IllegalArgumentException");
	/* If cls is NULL, an exception has already been thrown. */
	if(cls != NULL) {
		env->ThrowNew(cls, msg);
	}
	/* Free the local ref. */
	env->DeleteLocalRef(cls);
}
