/**
 * This is a SPECjvm2008 benchmark "crypto.signverify". This file was originally
 * Main.java.  The original source code in this file is a subject to
 * the following copyright:
 *
 * Copyright (c) 2008 Standard Performance Evaluation Corporation (SPEC)
 *               All rights reserved.
 *
 * This source code is provided as is, without any express or implied warranty.
 */

package rpg.modules.signverify;

import java.io.IOException;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;

import rpg.ModuleTimed;
import rpg.SessionStateHolder;
import rpg.StopBenchmarkException;
import rpg.Synchronized;
import rpg.modules.common.FileCache;
import rpg.modules.crypto.Util;

/**
 * Run this in single mode.
 * No multi threading for this benchmark right now.
 */
@Synchronized
public class SignverifyModule extends ModuleTimed {

	private static PublicKey rsaPub;
    private static PrivateKey rsaPriv;
    
    private static PublicKey dsaPub;
    private static PrivateKey dsaPriv;
    
    private final static int iterations = 10;
    
	private static boolean isSetUp = false;

	@Override
	public void init() {
		if(!isSetUp) {
			setupBenchmark();
			isSetUp = true;
		}
	}

	@Override
	protected void internalWork (SessionStateHolder sessionState) {
		runBenchmark();
	}

    public byte [] sign(byte[] indata, String algorithm, PrivateKey privKey) {
        
        try {
            Signature signature = Signature.getInstance(algorithm);
            signature.initSign(privKey);
            signature.update(indata);
            return signature.sign();
        } catch (Exception e) {
            throw new StopBenchmarkException("Exception in verify for " + algorithm + ".", e);
        }
    }
    
    public boolean verify(byte[] indata, String algorithm, byte [] signed, PublicKey pubKey) {
        
        try {
            
            Signature signature = Signature.getInstance(algorithm);
            signature.initVerify(pubKey);
            
            signature.update(indata);
            
            return signature.verify(signed);
            
        } catch (Exception e) {
            throw new StopBenchmarkException("Exception in verify for " + algorithm + ".", e);
        }
    }
    
    
    
    public void runSignVerify(byte[] indata, String algorithm, PrivateKey privKey, PublicKey pubKey) {
        
        byte [] signed = sign(indata, algorithm, privKey);
        // RPG_EDIT: output is ommited in rpg project
		/*boolean verification = */verify(indata, algorithm, signed, pubKey);
        
        /*if (verification) {
            spec.harness.Context.getOut().println(algorithm + " Verification PASS");
        } else {
            spec.harness.Context.getOut().println(algorithm + " Verification FAILED.");
        }*/
        
    }
    
    public void runBenchmark() {
        for (int i = 0; i < iterations; i++) {
			// RPG_EDIT: output is ommited in rpg project
            //Context.getOut().println("Iteration " + i + ".");
            runSignVerify(Util.getTestData(Util.TEST_DATA_4), "MD5withRSA", rsaPriv, rsaPub);
            runSignVerify(Util.getTestData(Util.TEST_DATA_4), "SHA1withRSA", rsaPriv, rsaPub);
            runSignVerify(Util.getTestData(Util.TEST_DATA_4), "SHA1withDSA", dsaPriv, dsaPub);
            runSignVerify(Util.getTestData(Util.TEST_DATA_4), "SHA256withRSA", rsaPriv, rsaPub);
            
            runSignVerify(Util.getTestData(Util.TEST_DATA_5), "MD5withRSA", rsaPriv, rsaPub);
            runSignVerify(Util.getTestData(Util.TEST_DATA_5), "SHA1withRSA", rsaPriv, rsaPub);
            runSignVerify(Util.getTestData(Util.TEST_DATA_5), "SHA1withDSA", dsaPriv, dsaPub);
            runSignVerify(Util.getTestData(Util.TEST_DATA_5), "SHA256withRSA", rsaPriv, rsaPub);
            
            runSignVerify(Util.getTestData(Util.TEST_DATA_6), "MD5withRSA", rsaPriv, rsaPub);
            runSignVerify(Util.getTestData(Util.TEST_DATA_6), "SHA1withRSA", rsaPriv, rsaPub);
            runSignVerify(Util.getTestData(Util.TEST_DATA_6), "SHA1withDSA", dsaPriv, dsaPub);
            runSignVerify(Util.getTestData(Util.TEST_DATA_6), "SHA256withRSA", rsaPriv, rsaPub);
        }
    }
    
    public static void setupBenchmark() {
        
        try {
            FileCache.getFileCache().loadFile(Util.TEST_DATA_4);
			//RPG_EDIT: It seems they load TEST_DATA_5 in both rsa and
			// signverify, how do they not get an exception I don't know
			// added check if it is already there
			if(!FileCache.getFileCache().hasFile(Util.TEST_DATA_5))
				FileCache.getFileCache().loadFile(Util.TEST_DATA_5);
            FileCache.getFileCache().loadFile(Util.TEST_DATA_6);
            
            KeyPairGenerator rsaKeyPairGen = KeyPairGenerator.getInstance("RSA");
            // 512, 768 and 1024 are commonly used
            rsaKeyPairGen.initialize(1024);
            
            KeyPair rsaKeyPair = rsaKeyPairGen.generateKeyPair();
            
            rsaPub = rsaKeyPair.getPublic();
            rsaPriv = rsaKeyPair.getPrivate();
            
            KeyPairGenerator dsaKeyPairGen = KeyPairGenerator.getInstance("DSA");
            dsaKeyPairGen.initialize(1024);
            
            KeyPair dsaKeyPair = dsaKeyPairGen.generateKeyPair();
            
            dsaPub = dsaKeyPair.getPublic();
            dsaPriv = dsaKeyPair.getPrivate();
        } catch (Exception e) {
            throw new StopBenchmarkException("Error in setup of crypto.aes." + e);
        }
    }
    
    public static void createTestData() throws IOException {
        Util.createRandomTestData(Util.TEST_DATA_4, 1024);
        Util.createRandomTestData(Util.TEST_DATA_5, 16384);
        Util.createRandomTestData(Util.TEST_DATA_6, 1048576);
    }
    
}
