#include "_module.h"

#include <math.h>
#include <cassert>

#include "exceptions.h"
#include "benchmarkfactory.h"

#include "memutils.h"
#include "parammacros.h"

using namespace std;
using namespace rip;

/** Constructor */
thrasher::thrasher (const std::string &name)
: module_default (name)
{
}

void thrasher::init (int count, ...) {
	assert (count == 0);

	s_access.reset(		sprovider.createSource(name, TRASHER_ACCESS_METHOD, true));
	s_stepsize.reset(	sprovider.createSource(name, TRASHER_STEPSIZE, true));
	s_memsize.reset(	sprovider.createSource(name, TRASHER_MEMSIZE, true));
	s_nopcount.reset(	sprovider.createSource(name, TRASHER_NOPCOUNT, true));
	s_maxnops.reset(	sprovider.createSource(name, TRASHER_MAXNOPSINFUNC, true));
	s_pointers.reset(	sprovider.createSource(name, TRASHER_POINTERS, true));
	s_write.reset(		sprovider.createSource(name, TRASHER_WRITE, true));
	s_steps.reset(		sprovider.createSource(name, TRASHER_STEPS));

	string access				= s_access->getStringValue ();
	size_t stepSize				= s_stepsize->getIntValue ();
	size_t memSize				= s_memsize->getIntValue ();
	size_t nopCount				= s_nopcount->getIntValue ();
	size_t maxNopsInFunction	= s_maxnops->getIntValue ();
	pointers					= s_pointers->getIntValue ();

	unsigned pageColors			= 0;

	// some tests
	size_t memSizeSep = memSize / pointers;

	if(memSizeSep < stepSize) {
		error("Thrasher: mem size is too small for one pointer");
		exit(EXIT_FAILURE);
	}

	if((unsigned int)stepSize < sizeof(void *)) {
		error("Thrasher: step size is smaller then pointer size");
		exit(EXIT_FAILURE);
	}

	if(stepSize % sizeof(void *) != 0) {
		error("Thrasher: step size is not multiple of pointer size");
		exit(EXIT_FAILURE);
	}

	if(memSize % sizeof(void *) != 0) {
		error("Thrasher: mem size is not multiple of pointer size");
		exit(EXIT_FAILURE);
	}

	// generate nop function
	nopFunc = createNopFunc(nopCount, maxNopsInFunction);

	// memory access init

	if(access == "linear") {

		mt = MemInit::linearmp(memSizeSep, stepSize, pointers, pageColors);
	}

	if(access == "random") {

		mt = MemInit::randommp(memSizeSep, stepSize, pointers, pageColors);
	}

	if(mt.getMemArea() == NULL) { // no memory was allocated
		throw new BenchmarkErrorException("access type \"" +
				access + "\" not implemented");
	}
}

void thrasher::deinit () {
	mt.free();
	destroyNopFunc(nopFunc);
}

/** Virtual destructor */
thrasher::~thrasher() {
}


/** Does a module's work - thrasher */
void thrasher::internal_work(int &session_state) {

	const int write					= s_write->getIntValue ();
	const size_t steps				= s_steps->getIntValue ();

	// allow these to be in registers
	const int pointers				= this->pointers;
	void * * * const trails 		= mt.getTrails();

	// tmp read vars
	// volatile is not needed here - gcc cannot optimize here
	void * * * p = new void * * [pointers];
	for(int i = 0; i < pointers; ++i) {
		p[i] = trails[i];
	}

	if(write) {
		// write
		for (size_t step = 0; step < steps; step++) {
			// do some nop instructions before mem access
			// access frequency control
			callNopFunc(nopFunc);

			for(int i = 0; i < pointers; ++i) {
				NEXT_ADDR_WITH_WRITE(p[i]);
			}
		}
	} else {
		// read
		for (size_t step = 0; step < steps; step++) {
			// do some nop instructions before mem access
			// access frequency control
			callNopFunc(nopFunc);

			for(int i = 0; i < pointers; ++i) {
				NEXT_ADDR(p[i]);
			}
		}
	}

	NO_OPTIM_POINTERS(p, pointers);

	delete[] p;
}
