#ifndef RIPLIB_H_
#define RIPLIB_H_

#include <string>
#include <iostream>
#include <sstream>
#include <cctype>
#include <cerrno>
#include <cstdio>
#include <stdint.h>
#include <algorithm>

#include <linux/unistd.h>
#include <sys/mman.h>

#include "fileconsts.h"
#include "exceptions.h"

namespace rip {

const int PAGE_SIZE_SHIFT = 12;
const int PAGE_SIZE = (1 << PAGE_SIZE_SHIFT);

/**
 * Returns thread id
 */
// thx http://www.redhat.com/archives/linux-cluster/2005-June/msg00127.html
inline pid_t getTID() {

	return syscall(__NR_gettid);
}

// modified version from http://en.wikipedia.org/wiki/RDTSC
extern "C" {
	inline uint64_t rdtsc() {
	uint32_t lo, hi;

#ifdef RIP_RDTSCP
	asm volatile ("rdtscp" : "=a" (lo), "=d" (hi) :: "%ecx");
#else
	asm volatile ("xorl %%eax,%%eax \n cpuid" ::: "%rax", "%rbx", "%rcx", "%rdx");
	asm volatile ("rdtsc" : "=a" (lo), "=d" (hi));
#endif

   return (uint64_t)hi << 32 | lo;
   }
}

/**
 * Fast random - Bill's generator
 *
 * But our is tuned for 32 bit :)
 */
// thx http://www.daniweb.com/code/snippet67.html
// http://softwarecommunity.intel.com/articles/eng/2978.htm
inline int fastRand(int seed) {

	seed = ((seed * 214013L + 2531011L) & 0x7FFFFFFF);
	return seed;
}

/**
 * Returns allocated memory of size "size" allocated by mmap
 *
 * @param size size that will be allocated
 * @param canExec pages with set executable bit
 * @param colors allocates memory area where virtual and physical pages have the
 *        same color
 */
void * mmapAlloc(size_t size, bool canExec = false, unsigned colors = 0);

/**
 * Frees memory allocated using mmapAlloc
 */
inline void mmapFree(void * start, size_t size) {

	if(start == NULL) {
		return;
	}

	int result = munmap(start, size);

	if(result == -1) {

		throw new SystemErrorException("Cannot free memory using munmap",
				errno);
	}
}

/**
 * Schedule thread to one cpu
 */
void cpuSched(int cpuNumber);

/**
 * Converts string with units (like K, M, ...) at the end to integer
 */
int64_t swutoi64(const std::string & str);

/**
 * itos function with minimal length - shorter strings are completed with 0
 */
std::string itosWithMinLength(int value, size_t length);

/**
 * Creates directory and subdirectories
 *
 * Same errors as std mkdir except errno 17 (- file exists) is ignored here
 */
int mkDirTree(const std::string & dirTree, mode_t mode);

/**
 * Adds path delimiter to the end of dir string if none is present
 */
std::string addDelimToDir(const std::string & dir);

/**
 * Returns string that has removed all chars charToRemove
 */
std::string removeAllChars(const std::string & str, char charToRemove);

/**
 * Returns string that has removed all white spaces
 */
std::string removeAllWhiteSpaces(const std::string & str);

/**
 * Removes white spaces from beginning and end of string
 */
std::string trim(const std::string & str);

}

#endif /*RIPLIB_H_*/
