#include "riplib.h"

#include <sys/stat.h>

#include "convert.h"
#include "colorallocator.h"

using namespace std;

namespace rip {

void cpuSched(int cpuNumber) {

	// cpu set
	cpu_set_t cpuSet;

	// init set
	CPU_ZERO(&cpuSet);

	// set set :)
	CPU_SET(cpuNumber, &cpuSet);

	if(sched_setaffinity(getTID(), sizeof(cpu_set_t), &cpuSet)) {

		throw new SystemErrorException(
				"Cannot schedule thread to cpu " + toString(cpuNumber), errno);
	}
}

int64_t swutoi64(const std::string & str) {

	const int64_t KILO = 1024;

	int64_t multiplier = 1;

	string strToConvert = str;

	switch(tolower(str[str.size() - 1])) {

	case 'k':
		multiplier = KILO;
		strToConvert = str.substr(0, str.size() - 1);
		break;

	case 'm':
		multiplier = KILO * KILO;
		strToConvert = str.substr(0, str.size() - 1);
		break;

	case 'g':
		multiplier = KILO * KILO * KILO;
		strToConvert = str.substr(0, str.size() - 1);
		break;

	case 't':
		multiplier = KILO * KILO * KILO * KILO;
		strToConvert = str.substr(0, str.size() - 1);
		break;

	default:
		// normal convert - no units
		multiplier = 1;
	}

	return stringTo<int64_t>(strToConvert) * multiplier;
}

std::string itosWithMinLength(int value, size_t minLength) {

	std::string result = toString(value);

	// not very effective but we don't expect much rounds here
	while(result.size() < minLength) {
		result = '0' + result;
	}

	return result;
}

int mkDirTree(const std::string & dirTree, mode_t mode) {

	size_t delim = 0;

	do {

		// get next path delim
		delim = dirTree.find(PATH_DELIM, delim + 1);

		string dirToCreate;

		// dir to create
		if(delim != string::npos) {
			// dir to delim
			dirToCreate = dirTree.substr(0, delim);
		}
		else {
			// whole dir
			dirToCreate = dirTree;
		}

		int mkdirRes = mkdir(dirToCreate.c_str(), mode);

		if(mkdirRes != 0 && errno != 17) { // errno 17 = file exists
			return mkdirRes;
		}

	} while(delim != string::npos);

	return 0;
}

std::string addDelimToDir(const std::string & dir) {

	string result = dir;

	if(result.rfind(PATH_DELIM) != result.size() - 1) {
		result += PATH_DELIM;
	}

	return result;
}

std::string removeAllChars(const std::string & str, char charToRemove) {

	const char SOMECHAR = 'a';

	string result = str;

	size_t charPos;
	charPos = result.find(charToRemove);
	while(charPos != string::npos) {

		// this removes char
		result.replace(charPos, 1, 0, SOMECHAR);
		charPos = result.find(charToRemove);

	}

	return result;
}

std::string removeAllWhiteSpaces(const std::string & str) {

	const char SPACE = ' ';
	const char TAB = '\t';
	const char NEWLINE = '\n';

	string noSpace = removeAllChars(str, SPACE);
	string noTab = removeAllChars(noSpace, TAB);
	string noNewline = removeAllChars(noTab, NEWLINE);

	return noNewline;
}

std::string trim(const std::string & str) {

	size_t noWhiteBegin = str.find_first_not_of(" \t\n");

	if(noWhiteBegin == string::npos) {
		return string();
	}

	size_t noWhiteEnd = str.find_last_not_of(" \t\n");

	return string(str, noWhiteBegin, noWhiteEnd - noWhiteBegin + 1);
}

void * mmapAlloc(size_t size, bool canExec, unsigned colors) {

	if(size == 0) {
		return NULL;
	}

	// set exec flag
	int execFlag = 0;
	if(canExec) {
		execFlag = PROT_EXEC;
	}

	// no colors path
	if(colors == 0) {

		void * result = mmap(NULL, size, PROT_READ | PROT_WRITE | execFlag,
				MAP_ANONYMOUS | MAP_PRIVATE, -1, 0);

		if( result == reinterpret_cast<void *>(-1) ) {

			throw new SystemErrorException("Cannot allocate memory using mmap",
					errno);
		}

		return result;
	}

	ColorAllocator ca(colors, canExec);

	return ca.cmmap(size);
}


}
