#ifndef NOPFUNCTION_H_
#define NOPFUNCTION_H_

#include "riplib.h"

namespace rip {

// NOTE: we need this really fast so no objects here
// otherwise this could be really nice class ;)

typedef void (* __NopFunctPtr)();

/**
 * Structure describing NOP function
 */
struct NopFunction {

	// function with maxNops length
	__NopFunctPtr fullNopF;

	// number of NOPs in result.fullNopF
	size_t fnfLength;

	// function shorter than maxNops
	__NopFunctPtr endNopF;

	// number of NOPs in endNopF
	size_t enfLength;

	// number of result.fullNopF loops that will be performed when callNopFunc
	// is called
	size_t fnfLoops;
};

// 1 is for ret at the end
static const int __NOP_FUNC_BUFFER_ADDITION = 1;

/**
 * Create NOP function
 *
 * @param nopCount how many NOPs in function
 * @return pointer to NOP function
 */
inline __NopFunctPtr __createNopFunc(size_t nopCount) {

	const int OPCODE_RET = 0xc3;
	const int OPCODE_NOP = 0x90;

	// NOP_FUNC_BUFFER_ADDITION is for ret at the end + set exec flag
	unsigned char * buff = static_cast<unsigned char *>(
			mmapAlloc(nopCount + __NOP_FUNC_BUFFER_ADDITION, true));

	// main function generation

	// add nop instruction to the buffer
	for (size_t i = 0; i < nopCount; ++i) {
		buff[i] = OPCODE_NOP;
	}
	// add ret instruction to the buffer
	buff[nopCount] = OPCODE_RET;

	return reinterpret_cast<__NopFunctPtr>(buff);
}

/**
 * Creates NOP function structure
 *
 * @param nopCount number of NOP counts that will be executed
 * @param maxNops max number of NOPs in one nop function
 * @return NOP function structure
 */
inline NopFunction createNopFunc(size_t nopCount, size_t maxNops) {

	NopFunction result;

	if(nopCount >= maxNops) {

		result.fnfLength = maxNops;
		result.fnfLoops = nopCount / maxNops;
		result.fullNopF = __createNopFunc(maxNops);

	}
	else {

		result.fnfLength = 0;
		result.fnfLoops = 0;
		result.fullNopF = NULL;

	}

	result.enfLength = nopCount % maxNops;

	result.endNopF = __createNopFunc(result.enfLength);

	return result;
}

/**
 * Executes NOP function
 *
 * @param nf NOP function structure
 */
inline void callNopFunc(const NopFunction & nf) {

	for(size_t i = 0; i < nf.fnfLoops; ++i) {
		nf.fullNopF();
	}

	nf.endNopF();
}

/**
 * Deallocates NOP function
 *
 * @param nopFunc NOP function structure
 */
inline void destroyNopFunc(NopFunction & nopFunc) {

	// NOP_FUNC_BUFFER_ADDITION is for ret at the end

	mmapFree(reinterpret_cast<void *>(nopFunc.fullNopF), // OK for NULL
			nopFunc.fnfLength + __NOP_FUNC_BUFFER_ADDITION);

	nopFunc.fullNopF = NULL;

	mmapFree(reinterpret_cast<void *>(nopFunc.endNopF),
				nopFunc.enfLength + __NOP_FUNC_BUFFER_ADDITION);

	nopFunc.endNopF = NULL;
}

}

#endif /*NOPFUNCTION_H_*/
