#ifndef MEMUTILS_H_
#define MEMUTILS_H_

#include "riplib.h"

namespace rip {

// ----------- Useful macros -----------

/**
 * Often used trick how to access next address in list and modify accessed
 * element
 */
#define NEXT_ADDR_WITH_WRITE(p)									\
																\
	/* 0 in lowest bit and 1 in every other place */			\
	const uintptr_t PTR_ALIGN_MASK = -2;						\
	const uintptr_t PTR_WRITE_MASK = 1;							\
																\
	/* get what is in pointer */								\
	uintptr_t readp = (uintptr_t)(*(p));						\
																\
	/* let's write to *p something else then it is there */		\
	/* in *p is address to another location so we cannot */		\
	/* modify it a lot */										\
	/* so we can change only lowest bit - address is aligned */	\
	/* if we clean that bit before access, all will be ok */	\
																\
	/* so if there (in lowest bit) is 0 this changes it */		\
	/* to 1 and vice versa */									\
	*(p) = (void *)(readp ^ PTR_WRITE_MASK);					\
																\
	/* this sets lowest bit to 0 - so we can access it */		\
	(p) = (void * *)(readp & PTR_ALIGN_MASK)

/**
 * Access next address
 */
#define NEXT_ADDR(p)				\
									\
	(p) = (void * *)(*(p))

/**
 * Code to avoid gcc to optimize accesses and cycles - multi pointer version
 */
#define NO_OPTIM(p)												\
	/* TRICK: we don't want gcc to optimize "p" mem access */	\
	/* so let it thing it was useful */							\
	void * * volatile nooptim;									\
	nooptim = (p)

/**
 * Code to avoid gcc to optimize accesses and cycles - multi pointer version
 */
#define NO_OPTIM_POINTERS(p, pointers)							\
																\
	/* TRICK: we don't want gcc to optimize "p" mem access */	\
	/* so let it thing it was useful */							\
																\
	void * * volatile * nooptim = new void * * [(pointers)];	\
																\
	for(int i = 0; i < pointers; ++i) {							\
		nooptim[i] = (p)[i];									\
	}															\
																\
	delete[] nooptim


// ----------- MemInit class -----------


static const size_t DEFAULT_FLUSH_CACHE_SIZE = 32 * 1024 * 1024; // 32 MB

/**
 * Holds allocated memory area and trails in it
 */
class MemTrails {

public:

	/**
	 * Create empty memory trail holder
	 */
	MemTrails() : mMemArea(NULL), mMemAreaSize(0), mTrails(NULL),
			mTrailsCount(0) { };

	/**
	 * Create and init memory trail holder
	 */
	MemTrails(char * memArea, size_t memAreaSize, void * * * trails, size_t trailsCount) :
			mMemArea(memArea), mMemAreaSize(memAreaSize), mTrails(trails),
			mTrailsCount(trailsCount) { };

	/**
	 * Free all memory trails and munmap memory
	 */
	void free() {

		delete[] mTrails;
		mmapFree(mMemArea, mMemAreaSize);
	}

	char * getMemArea() const {
		return mMemArea;
	}

	size_t getMemAreaSize() const {
		return mMemAreaSize;
	}

	void * * * getTrails() const {
		return mTrails;
	}

	void setTrails(void * * * pointers) {
		for (unsigned i = 0; i < mTrailsCount; i++) {
			mTrails[i] = pointers[i];
		}
	}

	size_t getTrailsCount() const {
		return mTrailsCount;
	}

private:

	char * mMemArea;
	size_t mMemAreaSize;
	void * * * mTrails;
	size_t mTrailsCount;
};

/**
 * Class for various memory initializations
 */
class MemInit {

public:

	/**
	 * Allocates memory with linear pointer trail in it
	 *
	 * !!! Result should be deallocated using mmapFree() !!!
	 *
	 * @param memSize memory size
	 * @param stepSize trail step size
	 * @param pageColors number of colors for colored allocation
	 *        (0 means normal allocation)
	 * @return first pointer in allocated memory
	 */
	static void * * linear(size_t memSize, size_t stepSize,
			size_t pageColors = 0) {

		const MemTrails & mt = linearmp(memSize, stepSize, 1, pageColors);

		// we can do this because of one pointer
		void * * result = (void * *) mt.getMemArea();

		// !TRICK! deallocate trails so result is only allocated object
		delete[] mt.getTrails();

		// return one pointer trail
		return result;
	}

	/**
	 * Allocates memory with linear pointer trails in it (multipointer version)
	 *
	 * !!! Call "result".free() to deallocate trail !!!
	 *
	 * @param memSize memory size
	 * @param stepSize trail step size
	 * @param pointersCount number of pointers (number of trails)
	 * @param pageColors number of colors for colored allocation
	 *        (0 means normal allocation)
	 * @param flushCacheSize size for flush cache
	 * @return MemTrails structure with trails in it
	 */
	static MemTrails linearmp(size_t memSize, size_t stepSize,
			size_t pointersCount,
			size_t pageColors = 0,
			size_t flushCacheSize = DEFAULT_FLUSH_CACHE_SIZE) {

		char * memArea = (char *) mmapAlloc(memSize, false, pageColors);
		void * * * trails = new void * *[pointersCount];

		size_t tAreaSize = memSize / pointersCount;

		for (size_t i = 0; i < pointersCount; ++i) {

			// trail area start
			char * tAreaStart = memArea + i * tAreaSize;
			// trail area end is after last allocated byte
			char * tAreaEnd = tAreaStart + tAreaSize;

			char * p = tAreaStart;

			// tAreaEnd - stepSize -> last address is for step to beginning
			while(p < tAreaEnd - stepSize) {
				*((void * *)p) = p + stepSize;
				p += stepSize;
			}

			// p is at most tAreaEnd - stepSize here

			// last address is for step to beginning
			*((void * *)p) = tAreaStart;

			trails[i] = (void * *) tAreaStart;
		}

		shuffle(trails, pointersCount, tAreaSize, stepSize);

		flushCache(flushCacheSize);

		return MemTrails(memArea, memSize, trails, pointersCount);
	}

	/**
	 * Allocates memory with random pointer trail in it
	 *
	 * !!! Result should be deallocated using mmapFree() !!!
	 *
	 * @param memSize memory size
	 * @param stepSize trail step size
	 * @param pageColors number of colors for colored allocation
	 *        (0 means normal allocation)
	 * @return first pointer in allocated memory
	 */
	static void * * random(size_t memSize, size_t stepSize,
			size_t pageColors = 0) {

		const MemTrails & mt = randommp(memSize, stepSize, 1, pageColors);

		// we can do this because of one pointer
		void * * result = (void * *) mt.getMemArea();

		// !TRICK! deallocate trails so result is only allocated object
		delete[] mt.getTrails();

		// return one pointer trail
		return result;
	}

	/**
	 * Allocates memory with random pointer trails in it (multipointer version)
	 *
	 * !!! Call "result".free() to deallocate trail !!!
	 *
	 * @param memSize memory size
	 * @param stepSize trail step size
	 * @param pointersCount number of pointers (number of trails)
	 * @param pageColors number of colors for colored allocation
	 *        (0 means normal allocation)
	 * @param flushCacheSize size for flush cache
	 * @return MemTrails structure with trails in it
	 */
	static MemTrails randommp(size_t memSize, size_t stepSize,
			size_t pointersCount,
			size_t pageColors = 0,
			size_t flushCacheSize = DEFAULT_FLUSH_CACHE_SIZE) {

		char * memArea = (char *) mmapAlloc(memSize, false, pageColors);
		void * * * trails = new void * *[pointersCount];

		size_t tAreaSize = memSize / pointersCount;

		for (size_t i = 0; i < pointersCount; ++i) {

			// trail area start
			char * tAreaStart = memArea + i * tAreaSize;

			// number of pointers in memSize
			size_t taPointersCount = tAreaSize / stepSize;

			// field of pointers
			void * * taPointers = new void *[taPointersCount];

			// get list of all available mem places
			for (size_t j = 0; j < taPointersCount; ++j) {
				taPointers[j] = tAreaStart + j * stepSize;
			}

			// this shuffles data with random access iterator - arrays too
			std::random_shuffle(taPointers, taPointers + taPointersCount);

			// get first from list...
			void * * p = (void * *)taPointers[0];

			// int i = 1 - ...and start from second
			for (size_t j = 1; j < taPointersCount; ++j) {
				*p = taPointers[j];
				p = (void * *)(*p);
			}

			// set last pointer to point to beginning
			*p = taPointers[0];

			delete[] taPointers;

			trails[i] = (void * *) tAreaStart;
		}

		shuffle(trails, pointersCount, tAreaSize, stepSize);

		flushCache(flushCacheSize);

		return MemTrails(memArea, memSize, trails, pointersCount);
	}

private:

	/**
	 * Flush cache content back to memory
	 *
	 * @param cacheSize cache size
	 */
	static void flushCache(size_t cacheSize) {

		// Get double size of last level cache size to evict more entries.
		// Last level cache should be physically indexed so double size will
		// help but something can still remain there. Life is tough.
		size_t fmemSize = 2 * cacheSize;

		// type for mem access
		typedef long flush_t;

		flush_t * fmem = (flush_t *)mmapAlloc(fmemSize);

		// number of steps in allocated memory
		size_t steps = fmemSize / sizeof(flush_t);

		// tmp var
		flush_t volatile p;

		// ignore cache line size and access whole memory
		for(size_t i = 0; i < steps; ++i) {

			p = fmem[i];
		}

		mmapFree(fmem, fmemSize);
	}

	/**
	 * Shuffle the pointers array for a better randomization and also
	 * changes a starting position in a pointer trail
	 *
	 * !!! Warning: shuffled pointers can not be used for area deallocation !!!
	 *
	 * @param trails trails to be shuffled
	 * @param pointersCount number of pointers in array
	 * @param tAreaSize trail area are size
	 * @param stepSize pointer trail block size
	 */
	static void shuffle(void * * * trails, int pointersCount, size_t tAreaSize,
			size_t stepSize) {

		// shuffle pointers in array
		std::random_shuffle(trails, trails + pointersCount);

		// randomize starting position in trails
		for(int i = 0; i < pointersCount; ++i) {

			// random shift
			// - calculate a random block where to begin
			// - and then get an offset for that block (in pointer arithmetic)
			size_t offset = rand() % (tAreaSize / stepSize);
			offset *= (stepSize / sizeof(void *));

			trails[i] = (void * *) (trails[i] + offset);
		}
	}

private:
	MemInit() { };
};

}

#endif /*MEMUTILS_H_*/
