#include "_module.h"

#include "exceptions.h"
#include "benchmarkfactory.h"
#include "convert.h"

#include "parammacros.h"

using namespace std;
using namespace rip;

#include <cassert>

/** Constructor */
memspeedmp::memspeedmp (const std::string &name)
: module_default (name)
{
}

void memspeedmp::init (int count, ...) {
	assert (count == 0);

	s_access.reset(		sprovider.createSource(name, MEMSPEEDMP_ACCESS_METHOD, true));
	s_stepsize.reset(	sprovider.createSource(name, MEMSPEEDMP_STEPSIZE, true));
	s_memsize.reset(	sprovider.createSource(name, MEMSPEEDMP_MEMSIZE, true));
	s_pointers.reset(	sprovider.createSource(name, MEMSPEEDMP_POINTERS, true));
	s_write.reset(		sprovider.createSource(name, MEMSPEEDMP_WRITE));

	// some tests
	string access				= s_access->getStringValue ();
	size_t memSize				= s_memsize->getIntValue ();
	size_t stepSize				= s_stepsize->getIntValue ();
	pointers					= s_pointers->getIntValue ();

	size_t memSizeSep = memSize / pointers;

	if(memSizeSep < stepSize) {
		error("Memspeedmp: mem size is too small for one pointer");
		exit(EXIT_FAILURE);
	}

	if((unsigned int)stepSize < sizeof(void *)) {
		error("Memspeedmp: step size is smaller then pointer size");
		exit(EXIT_FAILURE);
	}

	if(stepSize % sizeof(void *) != 0) {
		error("Memspeedmp: step size is not multiple of pointer size");
		exit(EXIT_FAILURE);
	}

	if(memSize % sizeof(void *) != 0) {
		error("Memspeedmp: mem size is not multiple of pointer size");
		exit(EXIT_FAILURE);
	}

	// memory access init

	if(access == "linear") {

		mt = MemInit::linearmp(memSizeSep, stepSize, pointers);
	}

	if(access == "random") {

		mt = MemInit::randommp(memSizeSep, stepSize, pointers);
	}

	if(mt.getMemArea() == NULL) { // no memory was allocated
		throw new BenchmarkErrorException("access type \"" +
				access + "\" not implemented");
	}

	/* how many pointer dereferences to sweep through the whole allocated memory */
	steps = memSizeSep / stepSize;
}

void memspeedmp::deinit () {
	mt.free();
}

/** Virtual destructor */
memspeedmp::~memspeedmp() {
}

void memspeedmp::internal_work(int &session_state) {

	const int write					= s_write->getIntValue ();

	// allow these to be in registers
	const size_t steps				= this->steps;
	const int pointers				= this->pointers;
	void * * * const trails 		= mt.getTrails();

	// tmp read vars
	// volatile is not needed here - gcc cannot optimize here
	void * * * p = new void * * [pointers];

	for(int i = 0; i < pointers; ++i) {
		p[i] = trails[i];
	}

	for (size_t step = 0; step < steps; step++) {

		if(write) {
			// write
			 {
				for(int i = 0; i < pointers; ++i) {
					NEXT_ADDR_WITH_WRITE(p[i]);
				}
			}
		}
		else {
			// read
			 {
				for(int i = 0; i < pointers; ++i) {
					NEXT_ADDR(p[i]);
				}
			}
		}
	}

	NO_OPTIM_POINTERS(p, pointers);

	delete[] p;
}
