#include "_module.h"

#include "atomic.h"
#include "exceptions.h"
#include "benchmarkfactory.h"

#include "memutils.h"
#include "parammacros.h"

using namespace std;
using namespace rip;

#include <cassert>

/** Constructor */
interleave::interleave (const std::string &name)
: module_default (name)
{
}

void interleave::init (int count, ...) {
	assert (count == 0);

	s_stepsize.reset(	sprovider.createSource (name, INTERLEAVE_STEPSIZE, true));
	s_blocksize.reset(	sprovider.createSource (name, INTERLEAVE_BLOCKSIZE, true));
	s_pointers.reset(	sprovider.createSource (name, INTERLEAVE_POINTERS, true));
	s_hops.reset(		sprovider.createSource (name, INTERLEAVE_HOPS));

	const unsigned MIN_BLOCK_SIZE = 64;

	blocksize 		= s_blocksize->getIntValue ();
	int stepsize	= s_stepsize->getIntValue ();
	pointers 		= s_pointers->getIntValue ();

	// Some sanity checks on the arguments
	if (blocksize < MIN_BLOCK_SIZE) {
		error("Interleave: block size is too small for one pointer");
		exit(EXIT_FAILURE);
	}
	if ((blocksize % stepsize) != 0) {
		error("Interleave: block size is not divisible by step size");
		exit(EXIT_FAILURE);
	}

	// Allocate and initialize the private memory block
	memoryBlock = MemInit::random (blocksize, stepsize);

	/* how many pointer dereferences to sweep through the whole allocated memory */
	steps = blocksize / stepsize;

	// Initialize the individual pointers
	fingers = new void ** [pointers];
	for (int i = 0 ; i < pointers ; i ++)
	{
		fingers [i] = memoryBlock;
	}
}

void interleave::deinit () {
	delete [] (fingers);

	mmapFree (memoryBlock, blocksize);
}

/** Virtual destructor */
interleave::~interleave() {
}


void interleave::internal_work (int &session_state)
{
	int hops		= s_hops->getIntValue ();

	const int pointers   = this->pointers;
	const unsigned steps = this->steps / hops;

	// Main benchmark loop
	for (size_t step = 0; step < steps; step++)
	{
		for (int iTrail = 0 ; iTrail < pointers ; iTrail ++)
		{
			for (int iHop = 0 ; iHop < hops ; iHop ++)
			{
				fingers [iTrail] = (void **) *fingers [iTrail];
			}
		}
	}
}
