#include "_module.h"

#include <unistd.h>
#include <pthread.h>

#include "memutils.h"
#include "parammacros.h"

using namespace std;
using namespace rip;

#include <cassert>

/** Constructor */
fft::fft (const std::string &name)
: module_default (name)
{
}

/** Init with no submodules */
void fft::init (int count, ...) {
	assert (count == 0);

	s_method.reset (sprovider.createSource(name, FFT_METHOD, true));
	s_bufsize.reset (sprovider.createSource(name, FFT_BUFSIZE, true));

	const string method   = s_method->getStringValue ();
	const size_t bufSize  = s_bufsize->getIntValue ();

	if (bufSize % sizeof(fftw_complex) != 0) {
		error("FFT: bufsize is not aligned to size of fft data type");
		exit(EXIT_FAILURE);
	}

	const int fftSize = bufSize / sizeof(fftw_complex);

	// memory buffer init

	fft_in = NULL;
	fft_out = NULL;

	if(method == "in-place") {

		fft_in = (fftw_complex*) fftw_malloc(bufSize);
		fft_out = fft_in;

	} else if (method == "separate") {

		fft_in = (fftw_complex*) fftw_malloc(bufSize);
		fft_out = (fftw_complex*) fftw_malloc(bufSize);

	} else {
		error("FFT: bad method parameter");
		exit(EXIT_FAILURE);
	}


	if (fft_in == NULL) {
		error("FFT: input buffer is null - allocation failure");
		exit(EXIT_FAILURE);
	}
	if (fft_out == NULL) {
		error("FFT: output buffer is null - allocation failure");
		exit(EXIT_FAILURE);
	}

	// fftw planning
	fft_plan = fftw_plan_dft_1d(fftSize, fft_in, fft_out, FFTW_FORWARD, FFTW_MEASURE);

	// initialize the buffer
	for (int i = 0; i < fftSize; i++) {
		fft_in[i][0] = i;
		fft_in[i][1] = 0;
	}
}

void fft::deinit () {
	// deallocation and cleanup
	if (fft_out != fft_in) {
		fftw_free (fft_out);
	}
	fftw_free (fft_in);
	fftw_destroy_plan (fft_plan);
	fftw_forget_wisdom();	
}

/** Virtual destructor */
fft::~fft() {
}

void fft::internal_work(int &session_state) {
	fftw_execute(fft_plan);
}
