#ifndef FACTORY_H_
#define FACTORY_H_

#include <string>
#include <map>
#include <cstdlib>

#include "logger.h"

namespace rip {

/**
 * Macro that registers class (Class) in factory (FactoryName) under id (RegID)
 */
#define REGISTER(FactoryName, Class, RegId)							\
																	\
	class FImpl##Class : public FactoryName::FactoryRegistrar {		\
																	\
	public:															\
		virtual FactoryName::TemplateParam * createObj() const {	\
																	\
			return new Class();										\
		}															\
																	\
	private:														\
		FImpl##Class() {											\
			FactoryName::reg(this, RegId);							\
		}															\
																	\
		static FImpl##Class * factory;								\
																	\
	};																\
																	\
	/* Create factory at global init */								\
	FImpl##Class * FImpl##Class::factory =							\
			new FImpl##Class()

/**
 * Templated factory design pattern
 */
template <class T>
class Factory {

public:

	typedef T TemplateParam;

	/**
	 * Base class used for Class registration
	 */
	class FactoryRegistrar {

	public:
		virtual ~FactoryRegistrar() { };

		/**
		 * Creates new object of registered class
		 *
		 * @return new object
		 */
		virtual T * createObj() const = 0;

	protected:
		FactoryRegistrar() { };
	};

	// all registered classes
	typedef std::map<std::string, FactoryRegistrar *> FactoryRegistrarMap;

	/**
	 * Register new class that implements FactoryRegistrar
	 *
	 * @param factoryRegistrar object that implements FactoryRegistrar
	 * @param regId class register id
	 */
	static void reg(FactoryRegistrar * factoryRegistrar,
			const std::string & regId) {

		typename FactoryRegistrarMap::const_iterator ite =
			getFactoryRegistrarMap().end();
		typename FactoryRegistrarMap::const_iterator factReg =
			getFactoryRegistrarMap().find(regId);

		// same type is already registered
		if(factReg != ite) {
			logE << "Name \"" + regId
					+ "\" is already registered"
					+ " - programming fault" << std::endl;

			exit(1);
		}

		getFactoryRegistrarMap()[regId] = factoryRegistrar;
	}

	/**
	 * Creates new object of registered class
	 *
	 * @param regId registered class id
	 * @return new object
	 */
	static T * create(const std::string & regId) {

		typename FactoryRegistrarMap::const_iterator ite =
			getFactoryRegistrarMap().end();
		typename FactoryRegistrarMap::const_iterator factReg =
			getFactoryRegistrarMap().find(regId);

		if(factReg == ite) {
			return NULL;
		}

		return factReg->second->createObj();
	}

protected:

	/**
	 * Holds all registered classes and deallocates them at the end
	 */
	class FactoryRegistrarMapHolder {

	public:

		/**
		 * Deallocates all registered classes
		 */
		~FactoryRegistrarMapHolder() {

			// free all FactoryRegistrars

			typename FactoryRegistrarMap::iterator it;
			typename FactoryRegistrarMap::iterator itEnd =
				factoryRegistrarMap.end();

			for(it = factoryRegistrarMap.begin(); it != itEnd; ++it) {
				delete it->second;
			}

			factoryRegistrarMap.clear();
		};

		FactoryRegistrarMap factoryRegistrarMap;
	};

	/**
	 * Returns map that holds all registered classes
	 *
	 * @return map that holds all registered classes
	 */
	static FactoryRegistrarMap & getFactoryRegistrarMap() {

		// this will init FactoryRegistrarMapHolder in time when we need it
		// if FactoryRegistrarMapHolder is only global static then it shall not
		// to be initialized when first register comes
		static FactoryRegistrarMapHolder factRegMapHolder;
		return factRegMapHolder.factoryRegistrarMap;
	}
};

}

#endif /*FACTORY_H_*/
