#ifndef EXCEPTIONS_H_
#define EXCEPTIONS_H_

#include <string>
#include <cstring>
#include <stdexcept>
#include <typeinfo>
#include <cerrno>
#include <cstdio>

namespace rip {

/**
 * Basic Exception
 */
class Exception : public std::exception
{
public:

	/**
	 * Create new exception with error message
	 *
	 * @param errorMsg error message
	 */
	Exception(const std::string & errorMsg)
	{
		mErrNumber = -1;
		mErrorMsg = errorMsg;
	}

	/**
	 * Create new exception with error message and errno
	 * Paste errno string message after exception message
	 *
	 * @param errorMsg error message
	 * @param errNumber errno
	 */
	Exception(const std::string & errorMsg, int errNumber)
	{
		mErrNumber = errNumber;

		char errBuff[BUFF_SIZE];
		char numberBuff[BUFF_SIZE];

		const std::string WITH_MSG_STR = " with message: ";

		sprintf(numberBuff, "%d", errNumber);

		mErrorMsg = errorMsg
			+ " (errno: " + numberBuff + ")"
			+ WITH_MSG_STR
			// strerror_r is thread safe
			+ std::string(strerror_r(errNumber, errBuff, BUFF_SIZE));
	}

	/**
	 * Virtual destructor
	 */
	virtual ~Exception() throw() {};

	/**
	 * Returns error message
	 *
	 * @return error message
	 */
	std::string errorMsg() const {

		return mErrorMsg;
	}

	/**
	 * Returns errno
	 *
	 * @return errno
	 */
	int errorNumber() const {

		return mErrNumber;
	}

	/**
	 * Returns error message (as std::exception)
	 *
	 * @return error message
	 */
	virtual const char* what() const throw() {
		return errorMsg().c_str();
	}

private:

	static const int BUFF_SIZE = 156;

	int mErrNumber;
	std::string mErrorMsg;
};

class FatalErrorException : public Exception {
public:
	FatalErrorException(const std::string & errorMsg)
		: Exception(errorMsg) { };
	FatalErrorException(const std::string & errorMsg, int errNumber)
		: Exception(errorMsg, errNumber) { };
};

class SystemErrorException : public Exception {
public:
	SystemErrorException(const std::string & errorMsg)
		: Exception(errorMsg) { };
	SystemErrorException(const std::string & errorMsg, int errNumber)
		: Exception(errorMsg, errNumber) { };
};

class InitErrorException : public Exception {
public:
	InitErrorException(const std::string & errorMsg)
		: Exception(errorMsg) { };
	InitErrorException(const std::string & errorMsg, int errNumber)
		: Exception(errorMsg, errNumber) { };
};

class PapiErrorException : public Exception {
public:
	PapiErrorException(const std::string & errorMsg)
		: Exception(errorMsg) { };
	PapiErrorException(const std::string & errorMsg, int errNumber)
		: Exception(errorMsg, errNumber) { };
};

class IOErrorException : public Exception {
public:
	IOErrorException(const std::string & errorMsg)
		: Exception(errorMsg) { };
	IOErrorException(const std::string & errorMsg, int errNumber)
		: Exception(errorMsg, errNumber) { };
};

class BadConfigException : public Exception {
public:
	BadConfigException(const std::string & errorMsg)
		: Exception(errorMsg) { };
	BadConfigException(const std::string & errorMsg, int errNumber)
		: Exception(errorMsg, errNumber) { };
};

class ResumeErrorException : public Exception {
public:
	ResumeErrorException(const std::string & errorMsg)
		: Exception(errorMsg) { };
	ResumeErrorException(const std::string & errorMsg, int errNumber)
		: Exception(errorMsg, errNumber) { };
};

class BadConversionException : public Exception {
public:
	BadConversionException(const std::string & errorMsg)
		: Exception(errorMsg) { };
		BadConversionException(const std::string & errorMsg, int errNumber)
		: Exception(errorMsg, errNumber) { };
};

class BenchmarkErrorException : public Exception {
public:
	BenchmarkErrorException(const std::string & errorMsg)
		: Exception(errorMsg) { };
	BenchmarkErrorException(const std::string & errorMsg, int errNumber)
		: Exception(errorMsg, errNumber) { };
};

}

#endif // EXCEPTIONS_H_
