#ifndef COLORALLOCATOR_H_
#define COLORALLOCATOR_H_

#include <map>

#include <vector>
#include <list>
#include <stdint.h>

namespace rip {

class ColorAllocator {

public:

	ColorAllocator(unsigned colors, bool canExec = false);

	virtual ~ColorAllocator();

	/**
	 * Maps pages with colored physical frames
	 *
	 * @param size size that will be allocated
	 * @return allocated memory
	 */
	void * cmmap(size_t size);

private:

	// to store page frame numbers
	typedef uintptr_t PageFrameNumber;

	// colored list types
	typedef std::list<void *> ColoredPagesList;
	typedef std::vector< ColoredPagesList > ColoredPages;

	/**
	 * store "virtual page number -> physical frame number" map of a contiguous
	 * memory area from /proc/self/pagemap
	 *
	 * !!! result should be deallocated using delete[] !!!
	 *
	 * @param area pointer to start of the area (must be page aligned)
	 * @param size size of the area in bytes (must be page aligned)
	 * @return pointer array of PageFrameNumber
	 */
	PageFrameNumber * virtualToPhysicalAddr(void * area, size_t size);


	/**
	 * Allocate new pages and add them to the color lists
	 *
	 * @param size memory size that will be allocated
	 */
	void allocNewPages(size_t size);

	/**
	 * Returns page with requested color
	 *
	 * @param color requested page color
	 * @return colored page
	 */
	void * getPage(unsigned color);

	// number of colors for allocation
	unsigned mColors;
	// page are executable
	bool mCanExec;
	// lists of colored pages
	ColoredPages mColoredPages;
};

}

#endif /* COLORALLOCATOR_H_ */
