#include "colorallocator.h"

#include <string>

#include <sys/types.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <stdlib.h>
#include <assert.h>

#include "exceptions.h"
#include "riplib.h"

using namespace std;

namespace rip {

ColorAllocator::ColorAllocator(unsigned colors, bool canExec) :
		mColors(colors), mCanExec(canExec) {

	mColoredPages.resize(mColors);
}

ColorAllocator::~ColorAllocator() {

	// deallocate (munmap) all unused pages
	for(unsigned color = 0; color < mColoredPages.size(); ++color) {

		ColoredPagesList::const_iterator it;
		ColoredPagesList::const_iterator itEnd = mColoredPages[color].end();

		for(it = mColoredPages[color].begin(); it != itEnd; ++it) {
			munmap(*it, PAGE_SIZE);
		}
	}
}

void * ColorAllocator::cmmap(size_t size) {

	// --- Map final (result) area,
	// realign it to match colors (starting color in final area will be 0)
    // and unmap unnecessary space

	// how many pages we need
	size_t sizeInPages = size / PAGE_SIZE;

	// allocate the final area, with extra pages for a realignment
	size_t nAASize = (sizeInPages + mColors - 1) * PAGE_SIZE;
	char * nonalignedArea = (char *) mmap(NULL, nAASize,
			PROT_READ | PROT_WRITE | (mCanExec ? PROT_EXEC : 0),
			MAP_PRIVATE | MAP_ANONYMOUS | MAP_NORESERVE, -1, 0);

	if(nonalignedArea == MAP_FAILED) {
		throw new SystemErrorException(
				"Cannot allocate memory using mmap",errno);
	}

	// realign the beginning
	size_t alignUnit = mColors * PAGE_SIZE;
	char * resultArea = (char *)
		// first add (alignUnit - PAGE_SIZE) so the alignment is done
		// in the allocated area
		((((uintptr_t)nonalignedArea + (alignUnit - PAGE_SIZE))
				// then align to the size of alignUnit
				/ alignUnit)
				// and get the result back as a pointer
				* alignUnit);

	// unmap unneeded pages from the beginning
	if(resultArea > nonalignedArea) {
		if(munmap(nonalignedArea, resultArea - nonalignedArea) != 0) {

			throw new SystemErrorException(
					"Cannot deallocate memory using munmap",errno);
		}
	}

	// unmap unneeded pages from the end
	char * resultAreaEnd = resultArea + size;
	char * nonalignedAreaEnd = nonalignedArea + nAASize;

	if(resultAreaEnd < nonalignedAreaEnd) {
		if(munmap(resultAreaEnd, nonalignedAreaEnd - resultAreaEnd) != 0) {
			throw new SystemErrorException(
					"Cannot deallocate memory using munmap",errno);
		}
	}

	// --- Create final area (remaping)

	// preallocate initial space - gatPage will run faster at the beginning
	allocNewPages(size);

	// get colored pages one by one and remap them to the end of the result area
	for(size_t i = 0; i < sizeInPages; ++i) {

		void * coloredPage = getPage(i % mColors);

		void * target = resultArea + i * PAGE_SIZE;

		if(mremap(coloredPage, PAGE_SIZE, PAGE_SIZE,
				MREMAP_FIXED | MREMAP_MAYMOVE, target) != target) {

			throw new SystemErrorException(
					"Cannot remap memory using mremap",errno);
		}
	}

	// --- Check that everything went as expected

	PageFrameNumber * resultAreaPFN = virtualToPhysicalAddr(resultArea, size);

	for (size_t i = 0; i < sizeInPages; i++) {

		// Bit 62 - page swapped
		const uint64_t swapMask = (uint64_t)1 << 62;
		if((resultAreaPFN[i] & swapMask) != 0) {
			throw new FatalErrorException(string(
					"ColorAllocator: Page was swapped during reallocations.")
					+ "Not enough memory for this test");
		}

		// proper page color vs. physical page color
		if(i % mColors != resultAreaPFN[i] % mColors) {
			throw new FatalErrorException(
					"ColorAllocator failed to allocate properly colored pages");
		}
	}

	delete[] resultAreaPFN;

	return resultArea;
}

ColorAllocator::PageFrameNumber * ColorAllocator::virtualToPhysicalAddr(
		void * area, size_t size) {

	/*
	 * Bits 0-54  page frame number (PFN) if present
	 * Bits 0-4   swap type if swapped
	 * Bits 5-54  swap offset if swapped
	 * Bits 55-60 page shift (page size = 1<<page shift)
	 * Bit  61    reserved for future use
	 * Bit  62    page swapped
	 */

	// entries in /proc/pid/pagemap are 64 bits
	const int PAGEMAP_ENTRY_SIZE = 8;
	// pagemap file name
	const string pagemapFN = "/proc/self/pagemap";

	// how many pages we need
	size_t sizeInPages = size / PAGE_SIZE;

	int pagemapFD = open(pagemapFN.c_str(), O_RDONLY | O_LARGEFILE);

	if(pagemapFD == -1) {
		throw new SystemErrorException("Cannot open pagemap file \""
				+ pagemapFN + "\"", errno);
	}

	uintptr_t start = ((uintptr_t) area) >> PAGE_SIZE_SHIFT;
	off64_t pagemapPos = start * PAGEMAP_ENTRY_SIZE;

	if(lseek64(pagemapFD, pagemapPos, SEEK_SET) != pagemapPos) {
		throw new SystemErrorException("Cannot seek in pagemap file \""
				+ pagemapFN + "\"", errno);
	}

	size_t pages = size >> PAGE_SIZE_SHIFT;
	uint64_t entry;

	PageFrameNumber * result = new PageFrameNumber[sizeInPages];

	for(size_t i = 0; i < pages; ++i) {

		if (read(pagemapFD, &entry, PAGEMAP_ENTRY_SIZE) != PAGEMAP_ENTRY_SIZE) {
			throw new SystemErrorException("Cannot read in pagemap file \""
					+ pagemapFN + "\"", errno);
		}

		// Bits 0-54 - page frame number (PFN) if present
		const uint64_t pfnMask = (int64_t)-1 >> (64 - 55);
		*(result + i) = (PageFrameNumber) (entry & pfnMask);
	}

	if (close(pagemapFD) == -1) {
		throw new SystemErrorException("Cannot close pagemap file \""
				+ pagemapFN + "\"", errno);
	}

	return result;
}

void ColorAllocator::allocNewPages(size_t size) {

	// get a memory area populated by physical pages
	char * newPages = (char *) mmap(NULL, size,
			PROT_READ | PROT_WRITE | (mCanExec ? PROT_EXEC : 0),
			MAP_PRIVATE | MAP_ANONYMOUS | MAP_POPULATE, -1, 0);

	if (newPages == MAP_FAILED) {
		throw new SystemErrorException(
				"Cannot allocate memory using mmap", errno);
	}

	// determine pfn of new allocated pages
	PageFrameNumber * newPagesPFN =
		virtualToPhysicalAddr(newPages, size);

	size_t sizeInPages = size / PAGE_SIZE;

	// add each page to the appropriate list
	for(size_t i = 0; i < sizeInPages; ++i) {

		unsigned color = newPagesPFN[i] % mColors;

		mColoredPages[color].push_back(newPages + i * PAGE_SIZE);
	}

	delete[] newPagesPFN;
}

void * ColorAllocator::getPage(unsigned color) {

	const size_t NEW_SIZE_ALLOC = 1024 * 4096;

	// alloc new pages until there is some page with needed color
	while(mColoredPages[color].size() == 0) {
		allocNewPages(NEW_SIZE_ALLOC);
	}

	void * result = mColoredPages[color].front();

	mColoredPages[color].pop_front();

	return result;
}

}
