#ifndef ATOMIC_H_
#define ATOMIC_H_

/**
 * Atomic operations that C can't guarantee us.  Useful for
 * resource counting etc..
 *
 * This is actually patched atomic
 * Var "counter" is volatile now
 *  - only read and write is affected
 *
 * Thx to linux kernel
 *
 * If you need more ops just look at...
 *
 * http://lxr.free-electrons.com/source/include/asm-x86/alternative_64.h
 * http://lxr.free-electrons.com/source/include/asm-x86/atomic_64.h
 * http://lxr.free-electrons.com/source/include/asm-x86/atomic.h
 *
 * ... or linux kernel src
 */

namespace rip {

/*
 * Make sure gcc doesn't try to be clever and move things around
 * on us. We need to use _exactly_ the address the user gave us,
 * not some alias that contains the same information.
 */
typedef struct { volatile int counter; } atomic_t;

/**
 * atomic_read - read atomic variable
 *
 * @param v pointer of type atomic_t
 *
 * Atomically reads the value of v.
 */
#define atomic_read(v)          ((v)->counter)

/**
 * atomic_set - set atomic variable
 *
 * @param v pointer of type atomic_t
 * @param i required value
 *
 * Atomically sets the value of v to i.
 */
#define atomic_set(v,i)         (((v)->counter) = (i))

/**
 * atomic_add - add integer to atomic variable
 *
 * @param i integer value to add
 * @param v pointer of type atomic_t
 *
 * Atomically adds i to v.
 */
static inline void atomic_add(int i, atomic_t *v)
{
	asm volatile(
			"lock addl %1,%0"
			:"=m" (v->counter)
			:"ir" (i), "m" (v->counter));
}

/**
 * atomic_sub - subtract the atomic variable
 *
 * @param i integer value to subtract
 * @param v pointer of type atomic_t
 *
 * Atomically subtracts i from v.
 */
static inline void atomic_sub(int i, atomic_t *v)
{
	asm volatile(
			"lock subl %1,%0"
			:"=m" (v->counter)
			:"ir" (i), "m" (v->counter));
}

/**
 * atomic_inc - increment atomic variable
 *
 * @param v pointer of type atomic_t
 *
 * Atomically increments v by 1.
 */
static inline void atomic_inc(atomic_t *v)
{
	asm volatile(
			"lock incl %0"
			:"=m" (v->counter)
			:"m" (v->counter));
}

/**
 * atomic_dec - decrement atomic variable
 *
 * @param v pointer of type atomic_t
 *
 * Atomically decrements v by 1.
 */
static inline void atomic_dec(atomic_t *v)
{
	asm volatile(
			"lock decl %0"
			:"=m" (v->counter)
			:"m" (v->counter));
}

}

#endif /*ATOMIC_H_*/
