/*
 * rand.h
 *
 *  Created on: Apr 1, 2010
 *      Author: caster
 */

#ifndef RPG_RAND_H_
#define RPG_RAND_H_

#include <math.h>

#include "common.h"

#include "../thirdparty/SFMT/SFMT.h"

//--------------------------------------------------------------------------
// Random number generator definitions

/// Random number generator lock.
/// Defined in config.cpp
extern volatile int rnd_lock;

inline void rand_init (rpg::rpg_uint_t _seed) {
	init_gen_rand(_seed);
}

inline rpg::rpg_uint_t rand_l (void) {
	while (__sync_lock_test_and_set (&rnd_lock, 1) == 1) {};

	volatile rpg::rpg_uint_t retval = gen_rand64();

	__sync_lock_release (&rnd_lock);

	return retval;
}

inline long double rand_ld(void) {
	/* generate random number between 0 and 1 */
	long double sum = 0.0;
	long double iter = 1.0 / 2.0;

	long rnd = rand_l();

	for (int i = 0; i < 32; i++) {
		if (rnd & 1) {
			sum += iter;
		}

		iter /= 2.0;
		rnd = rnd >> 1;
	}
	return sum;
}


inline long double rand_exp(long double lambda) {

	long double rnd = rand_ld();

	long double argument = 1.0 - rnd;
	return - logl(argument) * lambda;
}


/** Generates random numbers from geometric distribution.
 *
 * We use the variant Y (as denoted on Wikipedia) = number of failures
 * before the first success. This is more general, as the possible
 * outcomes include also zero, which is applicable for loops.
 * For loops, p is the probability of exiting the loop (thus including
 * the test before the first iteration).
 *
 * @param p The probability of success on each trial.
 */
inline long double rand_geom(long double p) {

	long double rnd = rand_ld();

	long double u = 1.0 - rnd;
	long double d = 1.0 - p;
	/* the log/log formula calculates X, we want Y = X - 1 */
	return ceill(logl(u) / logl(d) - 1.0);
}

#endif /* RPG_RAND_H_ */
