/** @file module.h
 * Module declarations.
 *
 * The standard module interface and other definitions
 * useful only to the modules but not to the generator,
 * such as handling timestamps.
 */ 

#ifndef __MODULE_H
#define __MODULE_H

#include <memory>
#include <string>
#include <vector>
#include <iostream>

#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>
#include <sys/time.h>

#include "common.h"
#include "config.h"


//--------------------------------------------------------------------------
// Externals

extern void modules_init (unsigned int threadcount);
extern void modules_destroy (void);
extern void lock_exclusive_mutex (void);
extern void unlock_exclusive_mutex (void);
extern int wait_barrier (void);

extern unsigned int max_measured_values;

//--------------------------------------------------------------------------
// Time

/** The type to store nanosecond measurements. */
typedef unsigned long long measure_time_t;

static const measure_time_t NANOSECONDS_IN_SEC = 1000000000ULL;


/** Conversion from struct timespec to measure_time_t. */
static inline
measure_time_t timespec_to_measure_time (const struct timespec & ts) {
	measure_time_t time = ts.tv_sec;
	time *= NANOSECONDS_IN_SEC;
	time += ts.tv_nsec;
	return time;
}

/** Conversion from measure_time_t to struct timespec. */
static inline
void measure_time_to_timespec (const measure_time_t time, struct timespec & ts) {
	ts.tv_sec = time / NANOSECONDS_IN_SEC;
	ts.tv_nsec = time % NANOSECONDS_IN_SEC;
}


/** Wrapper for clock_gettime. */
static inline
measure_time_t getClockTime (clockid_t clk_id) {

	struct timespec       clock;
	long                  retval;

	retval = clock_gettime (clk_id, &clock);
	if (retval != 0) {
		perror ("Cannot measure time");
		exit (EXIT_FAILURE);
	}

	return timespec_to_measure_time (clock);
}

static inline measure_time_t getMonotonicTime (void) { return getClockTime(CLOCK_MONOTONIC); }
static inline measure_time_t getThreadCpuTime (void) { return getClockTime(CLOCK_THREAD_CPUTIME_ID); }

class monotonic_timer {
private:
	measure_time_t start_time;
public:
	void start(void);
	measure_time_t stop(void);
};

class thread_cputime_timer {
private:
	measure_time_t start_time;
public:
	void start(void);
	measure_time_t stop(void);
};


//--------------------------------------------------------------------------
// Reliability

/// The type to store event counts.
typedef unsigned long long measure_count_t;

// WARNING The numerical ordering of the return values is used in code.

/// A return value indicating that the module has succeeded.
#define	RETURN_OK				0
/// A return value indicating that the module has encountered an error that is still latent.
#define RETURN_ERROR_LATENT		1
/// A return value indicating that the module has encountered an error that is already effective.
#define RETURN_ERROR_EFFECTIVE	2


//--------------------------------------------------------------------------
// Module Interfaces

/** Module base class.
 *
 * Provides management and measurement interface.
 *
 * WARNING The definition of the virtual methods of the module base class is
 * duplicated in the generator code, which needs to create an extra
 * type for each module instance. If the module interface changes,
 * the generator code needs to be changed as well.
 *
 * Each module instance is executed in a separate thread, no locking
 * should be necessary except when accessing static variables.
 */
class module_base
{
private:
	/// Pointer to the mutex synchronizing the internal_work method
	/// Initialized and used only for modules that call set_synchronized () returns true
	pthread_mutex_t * work_mutex;

protected:
	/// Make the module execution synchronized by a mutex
	void set_synchronized ();
	/// Determines whether execution of the module is synchronized
	inline bool is_synchronized () {
		return (work_mutex != NULL);
	}

public:
	/// Module instance name.
	std::string name;

	module_base (const std::string &name);
	virtual ~module_base ();

	/// Module initialization. Assumes to be protected by the exclusive mutex.
	virtual void init (int count, ...);
	/// Module deinitialization. Assumes to be protected by the exclusive mutex.
	virtual void deinit ();

	/** Module workload.
	 *
	 * The method executes the module workload. It is wrapped by the
	 * measured_work method, which collects the timestamps. Modules
	 * should implement their own workload.
	 */
	virtual void internal_work (int &session_state) = 0;

    /// The wrapper for workload that provides callbacks.
	void measured_work (int &session_state);

	/// The callback before the internal workload function is called.
	virtual bool before_internal_work (int &session_state);
	/// The callback after the internal workload function is called.
	virtual void after_internal_work (int &session_state);

	/// Measures the workload in an internal loop.
	void measure_isolated ();
    /// Prints collected measurements.
	virtual void print_measurements (const std::string &context);
	/// Clears collected measurements.
	virtual void clear_measurements ();
	/// Print module implementation-specific information useful for model transformations
	void print_configuration ();
};

typedef module_base* module_ptr;

/** Timed module base class.
 *
 * Extends the module base class with timing measurements.
 */
class module_timed : public module_base
{
private:
	/// Vector of operation measurements in monotonic and real time.
	std::vector<std::pair<measure_time_t, measure_time_t> > measured_times;

	/// The wall clock timer used to time an individual invocation.
	monotonic_timer mt;
	/// The thread clock timer used to time an individual invocation.
	thread_cputime_timer tt;

public:
	module_timed (const std::string &name);

    virtual void clear_measurements ();
	virtual void print_measurements (const std::string &context);

	virtual bool before_internal_work (int &session_state);
	virtual void after_internal_work (int &return_value);
};

/** Reliable module base class.
 *
 * Extends the timed module base class with reliability support.
 * The reliability support is trivial in that it always
 * returns whatever return value the internal work gave.
 */
class module_reliable : public module_timed
{
private:
	/// Counter of reliability return values.
	measure_count_t counted_result_ok;
	measure_count_t counted_result_error_latent;
	measure_count_t counted_result_error_effective;

	/// Determines whether session failure aborts execution.
	rpg::rpg_data_source_t bail_on_failure;

public:
	module_reliable (const std::string &name);

    virtual void clear_measurements ();
	virtual void print_measurements (const std::string &context);

	virtual bool before_internal_work (int &session_state);
	virtual void after_internal_work (int &session_state);
};

/** Unreliable module base class.
 *
 * Extends the reliable module base class with failure support.
 * The failure support can trigger failures regardless
 * of the return value of the internal work.
 */
class module_unreliable : public module_reliable
{
private:
    /// Determines whether the module contains a fault.
    bool internal_fault;
    /// Tracks whether the module caused an error.
    bool internal_error;

    rpg::rpg_data_source_t error_invocation_source;
    rpg::rpg_data_source_t error_termination_source;
    rpg::rpg_data_source_t error_internal_source;

    rpg::rpg_data_source_t failure_invocation_source;
    rpg::rpg_data_source_t failure_termination_source;
    rpg::rpg_data_source_t failure_internal_source;

public:
    module_unreliable (const std::string &name);

    virtual bool before_internal_work (int &session_state);
    virtual void after_internal_work (int &session_state);
};

typedef module_unreliable module_default;

#endif
