/** @file lang.h
 * Generated language specifics.
 *
 * This file provides interface for operations specific to the generated
 * language.
 */
#ifndef __LANG_H
#define __LANG_H

#include "common.h"

using namespace std;

namespace rpg {
	/** A singleton for operations specific to the generated language.
	 *
	 * For each possible generated language there is a class that inherits
	 * the interface specified by this abstract class. The program than uses
	 * this interface to do things like generating source code, without the need
	 * to be concerned which language is being generated. The singleton is
	 * accessed by a global variable lang_generator.
	 */
	class lang {
	public:
		/** Initializes the class for a specified language. */
		static bool init(rpg_str_t lang_name);

		/** Returns the name of an output filled template file. */
		virtual rpg_str_t getOutputMainFile() = 0;
		
		/** Returns the name of an input template file to be filled. */
		virtual rpg_str_t getInputTemplateFile() = 0;
		
		/* Various language generalizations for completing the template
		 * file. Each generates a method call on a specified module.
		 */
        virtual void generateDefinition(std::ostream& o, const string &indent, rpg_slot_module_t* module) = 0;
		virtual void generateDeclaration(std::ostream& o, const string &indent, rpg_slot_module_t* module) = 0;
		virtual void generateInitialization(std::ostream& o, const string &indent, rpg_slot_module_t* module) = 0;
		virtual void generateDeinitialization(std::ostream& o, const string &indent, rpg_slot_module_t* module) = 0;
        virtual void generateMeasure(std::ostream& o, const string &indent, rpg_slot_module_t* module) = 0;
        virtual void generateClear(std::ostream& o, const string &indent, rpg_slot_module_t* module) = 0;
		virtual void generateWork(std::ostream& o, const string &indent, rpg_slot_module_t* module) = 0;
		virtual void generatePrintTimes(std::ostream& o, const string &indent, rpg_slot_module_t* module) = 0;
		virtual void generatePrintConfig(std::ostream& o, const string &indent, rpg_slot_module_t* module) = 0;
	};

	
	/**
         * An implementation of lang class for the C++ language.
	 */
	class lang_cpp : public lang {
		rpg_str_t getOutputMainFile();
		rpg_str_t getInputTemplateFile();
        void generateDefinition(std::ostream& o, const string &indent, rpg_slot_module_t* module);
		void generateDeclaration(std::ostream& o, const string &indent, rpg_slot_module_t* module);
		void generateInitialization(std::ostream& o, const string &indent, rpg_slot_module_t* module);
		void generateDeinitialization(std::ostream& o, const string &indent, rpg_slot_module_t* module);
        void generateMeasure(std::ostream& o, const string &indent, rpg_slot_module_t* module);
        void generateClear(std::ostream& o, const string &indent, rpg_slot_module_t* module);
		void generateWork(std::ostream& o, const string &indent, rpg_slot_module_t* module);
		void generatePrintTimes(std::ostream& o, const string &indent, rpg_slot_module_t* module);
		void generatePrintConfig(std::ostream& o, const string &indent, rpg_slot_module_t* module);
	};
	
	/**
         * An implementation of lang class for the java language.
	 */
	class lang_java : public lang {
		rpg_str_t getOutputMainFile();
		rpg_str_t getInputTemplateFile();
        void generateDefinition(std::ostream& o, const string &indent, rpg_slot_module_t* module);
		void generateDeclaration(std::ostream& o, const string &indent, rpg_slot_module_t* module);
		void generateInitialization(std::ostream& o, const string &indent, rpg_slot_module_t* module);
		void generateDeinitialization(std::ostream& o, const string &indent, rpg_slot_module_t* module);
        void generateMeasure(std::ostream& o, const string &indent, rpg_slot_module_t* module);
        void generateClear(std::ostream& o, const string &indent, rpg_slot_module_t* module);
		void generateWork(std::ostream& o, const string &indent, rpg_slot_module_t* module);
		void generatePrintTimes(std::ostream& o, const string &indent, rpg_slot_module_t* module);
		void generatePrintConfig(std::ostream& o, const string &indent, rpg_slot_module_t* module);
	};

	/* TODO It may be useful to make lang more generic, instead of specific functions
	 * for initialization, work ... to make a generic function to generate code
	 * for some # directive, so there can be different directives for different
	 * languages. */

	/// A global reference to the singleton language generator.
	extern auto_ptr<lang> lang_generator;
}

/// C++ language name expected from the command line argument.
#define LANG_CPP_NAME "cpp"

/// Java language name expected from the command line argument.
#define LANG_JAVA_NAME "java"

/// Name of the input C++ template file.
#define LANG_INPUT_TEMPLATE_FILE_CPP		"main.tpl"

/// Name of the input JAVA template file.
#define LANG_INPUT_TEMPLATE_FILE_JAVA		"Worker.tpl"

/// Name of the output C++ source file of the generated application.
#define LANG_OUTPUT_MAIN_FILE_CPP		"main.cpp"

/// Name of the output Java source file of the generated application.
#define LANG_OUTPUT_MAIN_FILE_JAVA		"Worker.java"


#endif
