/** @file gener.h
 * Random program generator.
 *
 * This file provides interface to the main application generator.
 */ 
#ifndef __GENER_H
#define __GENER_H

#include "common.h"
#include "config.h"

#include <memory>
#include <ostream>
#include <set>

namespace rpg {

	/* forward declarations */
	class rpg_slot_module_t;
	class architecture_generator;

	/** Class representing module type */
	class rpg_formal_module_t {
	private:
		/** The generator that created this module. */
		architecture_generator& archGen;
		/** Type name of the module. */
		rpg_str_t name;
		/** Real class name of the module. */
		rpg_str_t classname;
		/** Cumulative probability. */
		rpg_uint_t probability;
	public:
		/** Slot number generator. */
		rpg_data_source_t slots;

		/** Expected duration of the module */
		rpg_uint_t expected_duration;

		/**
		 * Names of module configuration parameters that are used only in the generator,
		 * the rest of configuration parameters is processed for runtime configuration.
		 */
		static std::set<rpg_str_t> generator_parameters;

		rpg_formal_module_t(const rpg_str_t& name, const rpg_str_t& classname, dataSource* slots, 
			rpg_uint_t probability, architecture_generator& archGen);
		const rpg_str_t& getName(void);
		const rpg_str_t& getClassName(void);
		rpg_uint_t getSlots(void);
		rpg_uint_t getProbability(void);
		architecture_generator& getArchGen(void);
		rpg_slot_module_t* createModule(rpg_uint_t slot_count = -1);

		/** Create data source of a given configuration item of the module */
		dataSource * createItemDataSource (const rpg_str_t & item);
	};

	/** Class representing module instance. */
	class rpg_slot_module_t {
	private:
		/* parameter generator for both config and XML */
		void generateParam(std::ostream& o, const rpg_str_t& param_name, std::ostream& xmlparamdef);
	public:
		rpg_slot_module_t(rpg_formal_module_t& type, rpg_uint_t slots, rpg_str_t& name);
		~rpg_slot_module_t();
		/** Name of the instance of the module. */
		rpg_str_t name;
		/** Reference to the type class of the module. */
		rpg_formal_module_t& type;
		/** Number of slots of this module (number of submodules). */
		rpg_uint_t number_of_slots;
		/** Pointer to the array with submodules. */
		rpg_slot_module_t** slots;
		/** String containing XML definition of module parameters. */
		rpg_str_t xmlparamdef;
		/** How many times is the module used as a submodule (slot) of another module */
		rpg_uint_t refcount;

		/** Expected duration of the module */
		rpg_uint_t expected_duration;

		/* change number of slots */
		void change_slot_count (rpg_uint_t slot_count);

		/* definition generator */
		void generateXMLdef(std::ostream& o, bool oldStyleSlots);

		/* config generator */
		void generateConfig(std::ostream& o);
	};

	/** Base class for architecture generators */
	class architecture_generator {
	protected:
		/** Support for name generator functionality. */
		rpg_uint_t name_id;

		/** Total probability of the architectures. */
		rpg_uint_t arch_total_probability;
		/** Total probability of the ordinary modules. */
		rpg_uint_t module_total_probability;
		/** Probabilites on the different levels of recursion. */

		/** Module type storage. */
		std::vector<rpg_formal_module_t*> modules;
		/** Architecture type storage. */
		std::vector<rpg_formal_module_t*> architectures;

		/** Pointers to all generated modules */
		std::vector<rpg_slot_module_t*> generated_modules;

		/** Pointer to the root module of the generated architecture. */
		rpg_slot_module_t * root_module;

		/** Create a module randomly from given list of candidates */
		rpg_slot_module_t* create_module (const std::vector <rpg_formal_module_t*> & candidates);

		/** Create a module randomly from given list of candidates and given sum of all module probabilities */
		rpg_slot_module_t* create_module (const std::vector <rpg_formal_module_t*> & candidates, rpg_uint_t total_prob);

		/** Calculate and optionally set expected duration of an architectural module from its submodules */
		rpg_uint_t calculateExpectedDuration (rpg_slot_module_t * arch, rpg_int_t num_connected_slots = -1, bool set = true);

		/** Calculate expected durations of an architectural module from the already connected submodules, and a candidate module */
		rpg_uint_t calculateExpectedDuration (rpg_slot_module_t * arch, rpg_int_t num_connected_slots, rpg_slot_module_t * candidate);
	public:

		/** Whether to generate xml description as a recursive tree of modules (the old way).
		 *
		 * The new way is to reference modules that are called (from slots of architecture modules)
		 * by their id, instead of embedding them.
		 *
		 * Makes sense only for tree and test architectures. Can be removed once the transformation tools
		 * are updated.
		 */
		bool generateTreeXml;

		architecture_generator (void);
		virtual ~architecture_generator (void);

		bool load (void);

		rpg_str_t createModuleName (void);

		virtual void generateArchitecture (void) = 0;

		void generateSource (std::ostream& o, std::istream& i);

		void generateConfig (std::ostream& o);

		void generateXMLArchDef (std::ostream& o);

		static architecture_generator * create_instance (void);
	};

	class tree_architecture_generator : public architecture_generator {
	protected:
		/** Probabilities of architectural modules on different level of nesting */
		rpg_data_source_t arch_probabilities;
		/** Probability generator for choosing architectural modules. */
		rpg_data_source_t arch_prob_gen;

		rpg_slot_module_t* choose_random (rpg_uint_t level);
		rpg_slot_module_t* generate (rpg_uint_t level);

	public:
		tree_architecture_generator (void);

		virtual void generateArchitecture (void);
	};

	class test_architecture_generator : public architecture_generator {
	protected:
		/** Name of the root module in testing architecture */
		rpg_str_t test_arch_root;
		/** Name of the dummy module (used to inject into modules that need submodules) in testing architecture */
		rpg_str_t test_arch_dummy;
		/** Number of instances of each module in the testing architecture */
		rpg_uint_t test_arch_instances;

	public:
		test_architecture_generator (void);

		virtual void generateArchitecture (void);
	};

	class bottom_up_architecture_generator : public architecture_generator {
	private:
		/* Modules that have been connected already */
		std::vector <rpg_slot_module_t*> connected_modules;
		/* Modules that have not yet been connected */
		std::vector <rpg_slot_module_t*> unconnected_modules;

		/* How many worker modules to generate */
		rpg_uint_t workers_count;

		/* Boolean data source deciding on module reuse */
		rpg_data_source_t connected_module_reuse;

		/* Maximum expected duration for the whole architecture */
		rpg_uint_t maximum_expected_duration;

		void filterConnectedModules (rpg_slot_module_t * arch, rpg_uint_t slot_index, std::vector <rpg_slot_module_t *> & candidates);

		void filterCandidates (rpg_slot_module_t * arch, rpg_uint_t num_connected_slots, std::vector <rpg_slot_module_t *> & candidates,
				std::vector <rpg_slot_module_t *> & filtered_candidates);

		rpg_uint_t determineReusals (bool * module_reusals, rpg_uint_t num_slots);

		rpg_slot_module_t * createArchModule ();

	public:
		bottom_up_architecture_generator ();

		virtual void generateArchitecture ();
	};
}

/* sections in template file */

/// A literal used to expand component type definition in the template file.
#define GEN_DEFINE		"RPG_EXPAND_DEFINE"
/// A literal used to expand component instance declaration in the template file.
#define GEN_DECLARE		"RPG_EXPAND_DECLARE"
/// A literal used to expand component instance initialization in the template file.
#define GEN_INIT		"RPG_EXPAND_INIT"
/// A literal used to expand component instance deinitialization in the template file.
#define GEN_DEINIT		"RPG_EXPAND_DEINIT"
/// A literal used to expand isolated measurement and measurement dump in the template file.
#define GEN_MEASURE     "RPG_EXPAND_MEASURE"
/// A literal used to expand workload measurement init in the template file.
#define GEN_CLEAR       "RPG_EXPAND_CLEAR"
/// A literal used to expand workload invocation in the template file.
#define GEN_WORK		"RPG_EXPAND_WORK"
/// A literal used to expand workload measurement dump in the template file.
#define GEN_PRINTTIMES  "RPG_EXPAND_PRINTTIMES"
/// A literal used to expand component instance config printing in the template file.
#define GEN_PRINTCONFIG "RPG_EXPAND_PRINTCONFIG"


#define OUTPUT_XML_FILE			"main.xml"


/** Group/item separator in configuratino files. */
#define GEN_DOUBLEDOTS  "::"

/** Assign operator in configuration files. */
#define GEN_ASSIGN		" = "

/** Left bracket token. */
#define GEN_PARAMBEGIN  " ( "

/** Right bracket token */
#define GEN_PARAMEND	" ) "

/** Quotes x by quotation marks. Item must be printed by operator << to outputstream. */
#define GEN_QUOTE(x)    "\"" << x << "\""

/** Begin of the list in configuration files. */
#define GEN_LISTBEGIN   " list ( "

/** End of the list in configuration files */
#define GEN_LISTEND		" ) "

/** Separator token. */
#define GEN_SEP			", "

/** XML generator: generates XML item from the x. 
 Item must be printed by operator << to outputstream. */
#define XML_ITEM(x)		"<item>" << x << "</item>"

/** XML generator: generates XML attr of name and value, including leading space
 Item must be printed by operator << to outputstream. */
#define XML_ATTR(name,value)	" " name "=" << GEN_QUOTE(value)

/** Caption of global section in the generated configuration file. */
#define GEN_GLOBAL_CONFIGURATION_CAPTION "# global configuration"

/** Random seed string to generate into the target file. */
#define GEN_RANDOM_SEED "core::random-seed = start-time()"

#endif
