/** @file config.h
 * Configuration header.
 *
 * Definitions related to the configuration provider.
 */ 
#ifndef __CONFIG_H
#define __CONFIG_H

#include "common.h"
#include "parser.h"

#include <map>
#include <memory>
#include <utility>

/* class name definitions */

/** Class name of the void resource. */
#define CLASS_BASE_NAME               "void"
/** Class name of the integer resource. */
#define CLASS_INTEGER_NAME            "integer"
/** Class name of the float resource. */
#define CLASS_FLOAT_NAME              "float"
/** Class name of the boolean resource. */
#define CLASS_BOOLEAN_NAME            "boolean"
/** Class name of the string resource. */
#define CLASS_STRING_NAME             "string"
/** Class name of the integer list resource. */
#define CLASS_INTEGERLIST_NAME        "int-list"
/** Class name of the float list resource. */
#define CLASS_FLOATLIST_NAME          "float-list"
/** Class name of the string list resource. */
#define CLASS_STRINGLIST_NAME         "str-list"
/** Class name of the start time resource. */
#define CLASS_STARTTIME_NAME		  "start-time"
/** Class name of the time name resource. */
#define CLASS_TIME_NAME				  "time"
/** Class name of the random resource. */
#define CLASS_RANDOM_NAME			  "random"
/** Class name of the random range resource. */
#define CLASS_RANDOM_RANGE_NAME		  "random-range"
/** Class name of the random coin toss resource. */
#define CLASS_RANDOM_COIN_TOSS_NAME   "random-coin-toss"
/** Class name of the random int list resource. */
#define CLASS_RANDOM_INT_LIST_NAME	  "rand-int-list"
/** Class name of the random float list resource. */
#define CLASS_RANDOM_FLOAT_LIST_NAME  "rand-float-list"
/** Class name of the random string list resource. */
#define CLASS_RANDOM_STR_LIST_NAME	  "rand-str-list"
/** Class name of the interval resource. */
#define CLASS_RANDOM_INTERVAL_NAME	  "rand-interval"
/** Class name of the parameter resource. */
#define CLASS_PARAMETER				  "parameter"
/** Class name of the random geometric generator resource. */
#define CLASS_RND_GEOM_NAME			  "random-geometric"

/* common parameter definitions */

/** Name of the value parameter of a simple resource. */
#define CONFIG_COMMON_VALUE		    "value"

/** Minimum parameter name in various resources. */
#define CONFIG_COMMON_MIN				"min"
/** Maximum parameter name in various resources. */
#define CONFIG_COMMON_MAX				"max"
/** Values parameter name in various resources. */
#define CONFIG_COMMON_VALUES			"values"

/** Probability parameter name in various resources. */
#define CONFIG_COMMON_PROBABILITY       "probability"

/** Classname parameter name in the parameter resource. */
#define CONFIG_COMMON_CLASSNAME			"classname"

/** Mean value. */
#define CONFIG_COMMON_EX				"ex"

/* namespaces */

/// Default group prefix.
#define GROUP_DEFAULT                   "default"
/// Core group in the configuration file.
#define GROUP_CORE						"core"
/// Application group in the configuration file.
#define GROUP_APP						"app"


/* values */

/** Random seed configuration item. */
#define ITEM_RANDOM_SEED				"random-seed"
/** Random configuration item. */
#define ITEM_RANDOM     				"random"
/** Module path configuration item. */
#define ITEM_MODULE_PATH				"module-path"
/** Path separator configuration item. */
#define ITEM_PATH_SEPARATOR				"path-separator"
/** Modules configuration item. */
#define ITEM_MODULES					"modules"
/** Slot number configuration item. */
#define ITEM_SLOTS  					"slots"
/** Probability configuration item. */
#define ITEM_PROBABILITY    			"probability"
/** Expected duration of module configuration item. */
#define ITEM_EXPECTED_DURATION			"expected-duration"
/** Maximum expected duration configuration item. */
#define ITEM_MAX_EXPECTED_DURATION		"max-expected-duration"
/** Architecture probability configuration item. */
#define ITEM_ARCH_PROB 					"arch-prob"
/** Architecture probability generator configuration item. */
#define ITEM_ARCH_PROB_GEN				"arch-prob-gen"
/** Generation path configuration item. */
#define ITEM_GEN_PATH					"output-path"
/** Real class name of the module. */
#define ITEM_CLASS_NAME					"classname"

/** Name of the architecture generator to use (string) */
#define ITEM_ARCHITECTURE_GENERATOR		"architecture-generator"

/** Name of root module of testing architecture */
#define ITEM_TEST_ARCH_ROOT				"testing-architecture-root"
/** Name of dummy module of testing architecture */
#define ITEM_TEST_ARCH_DUMMY			"testing-architecture-dummy"
/** Number of instances of each module in the testing architecture */
#define ITEM_TEST_ARCH_INSTANCES		"testing-architecture-instances"

/** Number of worker modules to generate */
#define ITEM_WORKER_MODULES_COUNT		"worker-modules-count"
/** Bool data source deciding on connected module reuse */
#define ITEM_CONNECTED_MODULE_REUSE		"connected-module-reuse"

/** Minimum cycles configuration item. */
#define ITEM_MIN_CYCLES					"min-cycles"
/** Minimum time configuration item. */
#define ITEM_MIN_TIME					"min-time"
/** Number of rounds of module works before module time measure start. */
#define ITEM_FREE_ROUND_COUNT			"measure-warmup"
/** Number of measure cycles. */
#define ITEM_MEASURE					"measure-count"
/** Number of threads of application. */
#define ITEM_THREAD_COUNT				"thread-count"
/** Number of processor cores used by application (currently used detected
 *  and used as app output, setting has to be done manually by taskset).
 */
#define ITEM_CORES_COUNT				"cores-count"
/** Client wait time expected value. */
#define ITEM_CLIENT_WAIT_TIME_EX        "client-wait-time-usec-ex"
/** Number of clients. */
#define ITEM_CLIENT_COUNT				"client-count"
/** Time limit of the running application. */
#define ITEM_TIME_LIMIT 				"time-limit"
/** Maximum size of measured time storage */
#define ITEM_VALUES_STORAGE_MAX_SIZE	"values-storage-max-size"

/// A configuration item, specifies whether a module will skip execution during session failure.
#define CONFIG_BAIL_ON_FAILURE "bail-on-failure"

/// A configuration item, specifies what is the likelihood of module instance containing a fault.
#define CONFIG_FAULT_PROBABILITY "fault-probability"

/// A configuration item, specifies what is the chance of an invocation causing an error at invocation time.
#define CONFIG_ERROR_INVOCATION_PROBABILITY "error-invocation-probability"
/// A configuration item, specifies what is the chance of an invocation causing an error at termination time.
#define CONFIG_ERROR_TERMINATION_PROBABILITY "error-termination-probability"
/// A configuration item, specifies what is the chance of an invocation causing an internal error.
/// For sake of configuration simplicity, internal reliability is always handled at termination time.
#define CONFIG_ERROR_INTERNAL_PROBABILITY "error-internal-probability"

/// A configuration item, specifies what is the chance of an error turning into a failure at invocation time.
#define CONFIG_FAILURE_INVOCATION_PROBABILITY "failure-invocation-probability"
/// A configuration item, specifies what is the chance of an error turning into a failure at termination time.
#define CONFIG_FAILURE_TERMINATION_PROBABILITY "failure-termination-probability"
/// A configuration item, specifies what is the chance of an internal error turning into a failure.
/// For sake of configuration simplicity, internal reliability is always handled at termination time.
#define CONFIG_FAILURE_INTERNAL_PROBABILITY "failure-internal-probability"

/* Module-specific configs that are however used in the generators too */
#define CONFIG_LOOPCOUNT 		"loopcount"

#define ARCH_CLASS_LOOP	  		"loop"
#define ARCH_CLASS_BRANCH 		"branch"
#define ARCH_CLASS_SEQUENCE		"sequential"

/// Name of the generated application configuration file.
#define OUTPUT_CONFIG_FILE		"main.conf"

namespace rpg {

	/** Interval data type. */
	struct rpg_interval_t {
		/** Lower bound of interval. */
		rpg_uint_t min;
		/** Upper bound of interval. */
		rpg_uint_t max;
		rpg_interval_t(rpg_uint_t min, rpg_uint_t max);
	};

	/* Forward declarations */
	class dataSourceHolder;
	class dataSource;

	/* Type declarations */
	/** Map for holding the resource classes. */
	typedef std::map<rpg_str_t, dataSourceHolder*> rpg_holder_map_t;
	/** Map for holding the resource items */
	typedef std::map<rpg_str_t, dataSourceHolder*> rpg_source_item_map_t;
	/** Map for holding the groups of resources.*/
	typedef std::map<rpg_str_t, rpg_source_item_map_t*> rpg_source_group_map_t;

	/** Data source type. */
	typedef std::auto_ptr<dataSource> rpg_data_source_t;

	/** Data source base class.
	 *
	 * A data source class represents a value of a single configuration
	 * entry. Various access methods can be used to query the value.
	 * The base class is used as a parent for typed data source
	 * classes.
	 */
	class dataSource
	{
	private:
		const char *name;

	protected:
		rpg_str_t group;
		rpg_str_t item;

		/** Print configuration error and return the given exception
		 *
		 * It is done this way so one can write 'throw common_error (...)' and silence
		 * the warnings about missing return value.
		 */
		rpg_exception_t common_error (const char *cause, rpg_exception_t exc);

	public:
		dataSource(const dataSourceHolder *holder);
		virtual ~dataSource();

		void setIdentifiers (const rpg_str_t &group, const rpg_str_t &item);

		virtual rpg_int_t getIntValue ();
		virtual rpg_int_t getIntMin ();
		virtual rpg_int_t getIntMax ();
		virtual rpg_int_t getIntMean ();

		virtual rpg_float_t getFloatValue ();
		virtual rpg_float_t getFloatMin ();
		virtual rpg_float_t getFloatMax ();
		virtual rpg_float_t getFloatMean ();

		virtual bool getBoolValue ();

		virtual const rpg_str_t &getStringValue ();

		virtual rpg_uint_t array_size(void);
		virtual const rpg_str_t &operator[] (const rpg_uint_t index);

		virtual bool isVoid ();
		virtual bool isConstant ();

		virtual bool isInteger ();
		virtual bool isFloat ();
		virtual bool isString ();

		virtual bool isIntegerArray ();
		virtual bool isFloatArray ();
		virtual bool isStringArray ();

		virtual bool isInterval ();

		virtual const char *getName(void);

		virtual const rpg_list_int_t &access_int_list(void);
		virtual const rpg_list_str_t &access_str_list(void);
		virtual const rpg_list_float_t &access_float_list(void);

		virtual const rpg_list_int_t &access_interval_min_list(void);
		virtual const rpg_list_int_t &access_interval_max_list(void);
		virtual rpg_str_t toString(void);

		virtual void generateConfigAndXML (std::ostream &con, std::ostream &xml);
	};

	/** Holder of the data source */
	class dataSourceHolder {
	protected:
		/* Pointer to the dataClassProvider that created this */
		dataClassProvider * provider;
	public:
		dataSourceHolder (dataClassProvider * provider);

		virtual dataSourceHolder * create () = 0;
		void applyParam (const rpg_param_value_t & param);
		virtual void setParam(const rpg_str_t& name, const rpg_int_t& value);
		virtual void setParam(const rpg_str_t& name, const rpg_float_t& value);
		virtual void setParam(const rpg_str_t& name, const rpg_str_t& value);
		virtual void setParam(const rpg_str_t& name, const rpg_list_int_t& value);
		virtual void setParam(const rpg_str_t& name, const rpg_list_float_t& value);
		virtual void setParam(const rpg_str_t& name, const rpg_list_str_t& value);
		virtual const char* getName(void) const;
		virtual dataSource* createSource(void) = 0;

		virtual void applyParams(const rpg_params_t& params);
		virtual bool checkData(void);
		virtual ~dataSourceHolder();
	};

	/** Map of all data sources.
	 *
	 * The data source provider keeps a map of all the configuration items.
	 * The map is filled as the configuration file is being parsed.
	 * Each configuration item is represented by an instance
	 * of the corresponding data source holder class.
	 */
	class dataSourceProvider {
	private:
		rpg_source_group_map_t sources;
		dataClassProvider *classProvider;

		dataSourceHolder *locateSourceHolder (const rpg_str_t &group, const rpg_str_t &item);

	public:
		dataSourceProvider (dataClassProvider *classProvider);
		~dataSourceProvider ();

        dataSource *createSource(const rpg_str_t &group, const rpg_str_t &item, const bool checkConstant = false);

		void registerSource(const rpg_str_t& group, const rpg_str_t& item, dataSourceHolder* holder);

		template<typename param_t>
		void registerSource(const rpg_str_t& group, const rpg_str_t& item, const param_t & source);

		void dumpSources(void);
		rpg_source_item_map_t * getItemMap (const rpg_str_t & group);
	};


	/** Data class provider, provides classes of the data by name */
	class dataClassProvider {
	private:
		rpg_holder_map_t classes;
	public:
		dataClassProvider();
		~dataClassProvider();
		void registerClass(const rpg_str_t& className, dataSourceHolder* holder);
		void registerClass(dataSourceHolder* holder);
		dataSourceHolder* create(const rpg_str_t& className);
	};

}

extern rpg::dataClassProvider cprovider;
extern rpg::dataSourceProvider sprovider;

#endif
